/*****************************************************************************\
**                                                                           **
** PBX4Linux                                                                 **
**                                                                           **
**---------------------------------------------------------------------------**
** Copyright: Andreas Eversberg                                              **
**                                                                           **
** Port-mISDN-sip: SIP abstraction                                           **
**                                                                           **
\*****************************************************************************/ 


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <poll.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "main.h"

enum {
	READSTATE_HEADER,
	READSTATE_BODY,
};

int sip_bindsocket = -1;	/* socket to handle incomming connections */

static char *host2ip(char *buffer, unsigned long host)
{
	UPRINT(buffer, "%d.%d.%d.%d", host>>24, (host>>16)&0xff, (host>>8)&0xff, host&0xff);
	return(buffer);
}


/*
 * constructor
 */
Psip::Psip(int type, int sock, unsigned long host, char *portname, char *tone_dir) : PmISDN(type, /*mISDNport_sip*/NULL, portname, char *tone_dir)
{
	p_m_s_socket = sock;
	p_m_s_host = host;
	p_m_s_linebuffer[0] = '\0';
	p_m_s_readstate = READSTATE_HEADER;
	p_m_s_bodybuffer = NULL;
	p_m_s_bodylen = p_m_s_bodymax = 0;

	PDEBUG(DEBUG_SIP, "Created new SIPPort(%s).\n", portname);
}


/*
 * destructor
 */
Psip::~Psip()
{
	/* close socket */
	if (p_m_s_socket >= 0)
	{
		close(p_m_s_socket);
		fhuse--;
	}
}


/*
 * parses a line of the SIP request
 */
void Psip::parse_line(void)
{
}


/*
 * parses the body of the SIP request
 */
void Psip::parse_body(void)
{
}


/*
 * handler
 */
int Psip::handler(void)
{
	struct message *message;
	int len;
	char buffer[1024], *p, *nextline;
	int cause;

	/* handle destruction */
	if (p_m_delete/* && was noch...*/)
	{
		PDEBUG(DEBUG_SIP, "Psip(%s) doing pending release.\n", p_name);
		delete this;
		return(-1);
	}

	/* read from socket */
	len = recv(p_m_s_socket, buffer, sizeof(buffer)-1, 0);
	if (len == 0)
	{
		PDEBUG(DEBUG_SIP, "Psip(%s) socket has been closed.\n", p_name);
		cause = CAUSE_NORMAL;
		release:
		close(p_m_s_socket);
		fhuse--;
		p_m_s_socket = -1;
		while (p_epointlist)
		{
			message = message_create(p_serial, p_epointlist->epoint_id, PORT_TO_EPOINT, MESSAGE_RELEASE);
			message->param.disconnectinfo.cause = cause;
			message->param.disconnectinfo.location = LOCATION_PRIVATE_LOCAL;
			message_put(message);
			/* remove endpoint */
			free_epointlist(p_epointlist);
		}
		new_state(PORT_STATE_RELEASE);
		delete this;
		return(-1);
	}
	if (len < 0)
	{
		if (errno!=EWOULDBLOCK)
		{
			PDEBUG(DEBUG_SIP, "Psip(%s) socket failed with errno %d.\n", p_name, errno);
			cause = CAUSE_TEMPOFAIL;
			goto release;
		}
	}
	if (len > 0)
	{
		buffer[len] = '\0';
		p = buffer;
		lookagain:
		/* assemble data from socket */
		switch(p_m_s_readstate)
		{
			case READSTATE_HEADER:
			if ((nextline = strchr(p, '\n')))
			{
				/* we have a line termination */
				*nextline = '\0';
				SCAT((char *)p_m_s_linebuffer, (char *)p);
				PDEBUG(DEBUG_SIP, "Psip(%s) got complete SIP header line '%s'.\n", p_name, p_m_s_linebuffer);
				parse_line();
				len -= (nextline-p);
				p = nextline;
				goto lookagain;
			}
			SCAT((char *)p_m_s_linebuffer, (char *)p);
			break;

			case READSTATE_BODY:
			if (p_m_s_bodylen+len >= p_m_s_bodymax)
			{
				memcpy(p_m_s_bodybuffer + p_m_s_bodylen, p, p_m_s_bodymax-p_m_s_bodylen);
				PDEBUG(DEBUG_SIP, "Psip(%s) got complete SIP body.\n", p_name);
				parse_body();
				len -= (p_m_s_bodymax-p_m_s_bodylen);
				p += (p_m_s_bodymax-p_m_s_bodylen);
				if (len)
					goto lookagain;
			}
			memcpy(p_m_s_bodybuffer + p_m_s_bodylen, p, len);
			p_m_s_bodylen += len;
			break;
		}
		return(1);
	}

	return(PmISDN::handler());
}


/*
 * endpoint sends messages to the port
 */
int Psip::message_epoint(unsigned long epoint_id, int message_id, union parameter *param)
{
	struct message *message;

	if (PmISDN::message_epoint(epoint_id, message_id, param))
		return(1);

	switch(message_id)
	{
#if 0
		case MESSAGE_INFORMATION: /* overlap dialing */
		PDEBUG(DEBUG_SIP, "Psip(%s) received dialing info '%s'.\n", p_name, param->information.number);
		message_information(epoint_id, message_id, param);
		break;

		case MESSAGE_SETUP: /* dial-out command received from epoint */
		PDEBUG((DEBUG_SIP | DEBUG_BCHANNEL), "Psip(%s) SIP port received setup from '%s' to '%s'\n", p_name, param->setup.callerinfo.id, param->setup.dialinginfo.number);
		message_setup(epoint_id, message_id, param);
		break;

		case MESSAGE_NOTIFY: /* display and notifications */
		PDEBUG(DEBUG_SIP, "Psip(%s) SIP port with (caller id %s) received notification: display='%s' notify=%d phone=%s\n", p_name, p_callerinfo.id, param->notifyinfo.display, param->notifyinfo.notify, param->notifyinfo.id);
		message_notify(epoint_id, message_id, param);
		break;

		case MESSAGE_FACILITY: /* facility message */
		PDEBUG(DEBUG_SIP, "Psip(%s) SIP port with (caller id %s) received facility.\n", p_name, p_callerinfo.id);
		message_facility(epoint_id, message_id, param);
		break;

		case MESSAGE_OVERLAP: /* more information is needed */
		message_overlap(epoint_id, message_id, param);
		break;

		case MESSAGE_PROCEEDING: /* call of endpoint is proceeding */
		PDEBUG(DEBUG_SIP, "Psip(%s) SIP port with (caller id %s) received proceeding\n", p_name, p_callerinfo.id);
		message_proceeding(epoint_id, message_id, param);
		break;

		case MESSAGE_ALERTING: /* call of endpoint is ringing */
		PDEBUG(DEBUG_SIP, "Psip(%s) SIP port with (caller id %s) received alerting\n", p_name, p_callerinfo.id);
		message_alerting(epoint_id, message_id, param);
		break;

		case MESSAGE_CONNECT: /* call of endpoint is connected */
		PDEBUG(DEBUG_SIP, "Psip(%s) SIP port with (caller id %s) received connect\n", p_name, p_callerinfo.id);
		message_connect(epoint_id, message_id, param);
		break;

		case MESSAGE_DISCONNECT: /* call has been disconnected */
		PDEBUG(DEBUG_SIP, "Psip(%s) SIP port with (caller id %s) received disconnect cause=%d\n", p_name, p_callerinfo.id, param->disconnectinfo.cause);
		message_disconnect(epoint_id, message_id, param);
		break;

		case MESSAGE_RELEASE: /* release SIP port */
		PDEBUG(DEBUG_SIP, "Psip(%s) SIP port with (caller id %s) received release cause=%d\n", p_name, p_callerinfo.id, param->disconnectinfo.cause);
		message_release(epoint_id, message_id, param);
		break;
#else
#warning messages disabled
#endif

		default:
		PDEBUG(DEBUG_SIP, "Psip(%s) SIP port with (caller id %s) received a wrong message: %d\n", p_name, p_callerinfo.id, message);
	}

	return(1);
}

int sip_handler(void)
{
	int notidle = 0;
	int sock;
	struct sockaddr_in sock_address;
	socklen_t length_address = sizeof(sock_address);
	unsigned long host;
	char buffer[16];
	class Psip *psip;
	unsigned long on = 1;

	/* a new connection will be handled */
	if ((sock = accept(sip_bindsocket,(struct sockaddr *)&sock_address,&length_address))>=0)
	{
		notidle = 1;
		host = ntohl((unsigned int)sock_address.sin_addr.s_addr);
		PDEBUG(DEBUG_SIP, "Incomming connection from host %s\n", host2ip(buffer, host));
		fhuse++;
		if(ioctl(sock, FIONBIO, (unsigned char *)(&on)) < 0)
		{
			PERROR("Failed to set incomming socket into nonblocking IO.\n");
			close(sock);
			fhuse--;
			return(1);
		}
		if (!(psip = new Psip(PORT_TYPE_SIP_IN, sock, host, host2ip(buffer, host))))
		{
			PERROR("Failed to create SIP object.\n");
			close(sock);
			fhuse--;
			return(1);
		}
	}

	return(notidle);
}

int sip_init(void)
{
	struct sockaddr_in bindsocket_address;
	unsigned long on = 1;

	/* create bind socket */
	if ((sip_bindsocket = socket(PF_INET, SOCK_STREAM, 0)) < 0)
	{
		PERROR("Failed to crate bind socket.\n");
		return(0);
	}
	fhuse++;
	if(ioctl(sip_bindsocket,FIONBIO,(unsigned char *)(&on))<0)
	{
		PERROR("Failed to set bind socket into nonblocking IO.\n");
		socket_failed:
		close(sip_bindsocket);
		sip_bindsocket = -1;
		fhuse--;
		return(0);
	}
	if(setsockopt(sip_bindsocket,SOL_SOCKET,SO_REUSEADDR,(unsigned char *)(&on),sizeof(on))<0)
	{
		PERROR("Failed to set bind socket SO_REUSEADDR.\n");
		goto socket_failed;
	}
	memset(&bindsocket_address, 0, sizeof(bindsocket_address));
	bindsocket_address.sin_family = AF_INET;
	bindsocket_address.sin_port = htons(options.sip_port);
	if (bind(sip_bindsocket,(struct sockaddr *)&bindsocket_address, sizeof(bindsocket_address))<0)
	{
		PDEBUG(DEBUG_SIP, "Failed to bind SIP socket to port %d. (errno=%d)\n", options.sip_port, errno);
		goto socket_failed;
	}
	if (listen(sip_bindsocket, options.sip_maxqueue))
	{
		PERROR("Failed to set socket into listen mode\n");
		goto socket_failed;
	}
	return(1);
}

void sip_cleanup(void)
{
	if (sip_bindsocket >= 0)
	{
		close(sip_bindsocket);
		fhuse--;
		sip_bindsocket = -1;
	}
}



