/*****************************************************************************\
**                                                                           **
** PBX4Linux                                                                 **
**                                                                           **
**---------------------------------------------------------------------------**
** Copyright: Andreas Eversberg + Joerg Habenicht                            **
**                                                                           **
** vector header file                                                        **
** contains interface for generic arrays of objects (just like STL)          **
**                                                                           **
\*****************************************************************************/ 

#ifndef RULE_H
#define RULE_H

#include "vector.h"

#include <stdio.h>




/* in fact, all these Value classes are only to enjoy
 * the route_param and route_cond structs.
 * Can we delete them later on? */

class Value
{
  /* one of: VALUE_TYPE_NULL, VALUE_TYPE_INTEGER, VALUE_TYPE_INTEGER_RANGE,
   * VALUE_TYPE_STRING, VALUE_TYPE_STRING_RANGE */
  int classification;
  
protected:
  Value(int clas);
public:
  Value();
  virtual ~Value(){};
  int get_classification() const;
  virtual struct route_param *fill_route_param(struct route_param *fill) const;
  virtual struct route_cond *fill_route_cond(struct route_cond *fill) const;
};

class IntegerValue: public Value
{
protected:
  IntegerValue(int clas);
public:
  int value;
  
  IntegerValue();
  virtual ~IntegerValue(){};
  //int getValue() const;
  virtual struct route_param *fill_route_param(struct route_param *fill) const;
  virtual struct route_cond *fill_route_cond(struct route_cond *fill) const;
};

class IntegerRangeValue: public IntegerValue
{
public:
  int value_to;
  
  IntegerRangeValue();
  virtual ~IntegerRangeValue(){};
  //int getValueTo() const;
  virtual struct route_param *fill_route_param(struct route_param *fill) const;
  virtual struct route_cond *fill_route_cond(struct route_cond *fill) const;
};

class StringValue: public Value
{
protected:
  StringValue(int clas);
public:
  char *value;
  bool autodelete_value; // delete value upon destruction ?
  
  StringValue();
  virtual ~StringValue();
  
  void setvalue(const char *str, int len = -1); // makes a deep copy of str
  int parse(const char *str); // parses a string ("[a-z]",) out of str, returns position of next unparsed char
  virtual struct route_param *fill_route_param(struct route_param *fill) const;
  virtual struct route_cond *fill_route_cond(struct route_cond *fill) const;
};

class StringRangeValue: public StringValue
{
public:
  char *value_to;
  
  StringRangeValue();
  virtual ~StringRangeValue();

  void setvalueto(const char *str, int len = -1); // makes a deep copy of str
  int parse(const char *str); // parses a string range ("[a-z]-[a-z]",) out of str, returns position of next unparsed char
  virtual struct route_param *fill_route_param(struct route_param *fill) const;
  virtual struct route_cond *fill_route_cond(struct route_cond *fill) const;
};

typedef AutodeleteVector<Value *>	ValueVector;


/* this one provides basic functions to parse the text.
 * it is used in all parsing classes */
class Parser
{
  /* test, if this character after the token is an allowed one
   * i.e. spaces ' ', ':' or string end '\0' */
  static bool is_allowed_condition_tokenend(char c);
  /* test, if this character after the token is an allowed one
   * i.e. spaces ' ' or string end '\0' */
  static bool is_allowed_action_tokenend(char c);

  /* fills the content of "parse" into valuevector.
   * decides on its own, whether the content is numeric or string.
   * content may be "abc", "abc,abc", "abc-def,ghi" or combinations of.
   * test for correctness of the content has to be done beforehand */
  static bool fill_valuevector(const char *parse, int len, ValueVector &valuevector);
  
public:
  /* gets the whole line and position to start parsing.
   * parses arguments, i.e. the token start with '='.
   * if parsing is successful, the length of the token is provided in "tokenlen", start in "tokenstart"
   * returns: 0 if success, -1 if error */
  static int parse_string_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector, bool is_range_possible = false);
  static int parse_int_argument   (const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector);
  static int parse_yesno_argument (const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector);
  static int parse_service_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector);
  static int parse_diversion_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector);
  static int parse_conntype_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector);
  static int parse_destination_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector);
  static int parse_time_argument  (const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector);
  static int parse_mday_argument  (const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector);
  static int parse_month_argument (const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector);
  static int parse_year_argument  (const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector);
  static int parse_wday_argument  (const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector);
  static int parse_interface_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector);
};


/* this one detects and contains the condition part of a rule
 * i.e. "extern :" , "dialing=12345 :", "default :" or " :" */
class Condition: protected Parser
{
  int type;
  int index;
  int match;
  char *args;
  ValueVector vv;
  
public:
  Condition();
  ~Condition();
  
  const char * const get_txt() const;
  //int parse(FILE *route_conf, const char *fname, int *lineno);
  int parse(const char *line, const char *fname, int lineno, int *parsepos);
  int debug_out(FILE *outf) const;	// writes the content of the rules to outf
  
  /* compatibility service functions */
  
  /* returns a pointer to malloc()ed memory for a complete ruleset.
   * calls the Ruler class.
   * NOTE: you have to free() the memory yourself! */
  struct route_cond *get_struct() const;
};




class ActionParameter: protected Parser
{
public:
  ActionParameter();
  ~ActionParameter();
  
  const char *get_txt() const;
  const char *get_arg() const;

  int route_param_index; // only for "get_struct()"  
  unsigned long long id; // type from struct param_defs.id
  char *args;
  ValueVector vv;

  int debug_out(FILE *outf) const;	// writes the content of the rules to outf
  
  /* compatibility service functions */
  
  /* returns a pointer to malloc()ed memory for a complete ruleset.
   * calls the Ruler class.
   * NOTE: you have to free() the memory yourself! */
  struct route_param *get_struct() const;
};




typedef AutodeleteVector<ActionParameter *>	ActionParameterVector;
/* this one contains the action part of a rule
 * i.e. "intern extension=4321", "goto extern" */
class Action: protected Parser
{
  int id;
  int action_defs_index;
  ActionParameterVector	apv;
  
public:
  bool is_parsetest; // ==true, if unshure, whether this line is only action or full rule. stops error output
  
  Action();
  
  const char * const get_txt() const;
  //int parse(FILE *route_conf, const char *fname, int *lineno);
  int parse(const char *line, const char *fname, int lineno, int *parsepos);
  int debug_out(FILE *outf) const;	// writes the content of the rules to outf
  
  /* compatibility service functions */
  
  /* returns a pointer to malloc()ed memory for a complete ruleset.
   * calls the Ruler class.
   * NOTE: you have to free() the memory yourself! */
  struct route_action *get_struct() const;
};


typedef AutodeleteVector<Condition *>	ConditionVector;
typedef AutodeleteVector<Action *>	ActionVector;
/* this one detects and contains a rule
 * i.e. "dialing=12345 : intern extension=4321\n"
 * "default: goto extern\n" */
class Rule
{
  ConditionVector	cv;
  ActionVector		av;
  
public:
  int	parse(FILE *route_conf, const char *fname, int *lineno); // parse FILE until next ruleset or EOF. return -1 if parse error, 0 if correct, 1 if EOF
  int debug_out(FILE *outf) const;	// writes the content of the rules to outf

  /* compatibility service functions */
  
  /* returns a pointer to malloc()ed memory for a complete ruleset.
   * calls the Ruler class.
   * NOTE: you have to free() the memory yourself! */
  struct route_rule *get_struct() const;
};





typedef AutodeleteVector<Rule*>	RuleVector;
/* this one detects and contains a rule set.
 * i.e. "[main]\n" */
class Ruleset
{
  char *name;
  RuleVector rulevector;
  
public:
  Ruleset();
  ~Ruleset();
  
  int	parse(FILE *route_conf, const char *fname, int *line); // parse FILE until next ruleset or EOF. return -1 if parse error, 0 if correct, 1 if EOF
  const char * const get_name() const;
  int debug_out(FILE *outf) const;	// writes the content of the rules to outf

  /* compatibility service functions */
  
  /* returns a pointer to malloc()ed memory for a complete ruleset.
   * calls the Ruler class.
   * NOTE: you have to free() the memory yourself! */
  struct route_ruleset *get_struct() const;
};

typedef AutodeleteVector<Ruleset*>	RulesetVector;

class RulesetRoot
{
  static RulesetVector ruleset;
public:
  static int parse_all(FILE *route_conf, const char *fname); // parse the whole file into a vector of rulesets
  static const RulesetVector &get_vector(); // get the parsed rulesets, needs previous call to parse_all()
  static int debug_out(FILE *outf);	// writes the content of the rules to outf
  
  static struct route_ruleset *get_struct(); // hands back the root of a ruleset tree
};


/* module test function */
int test_read_config(const char *fname);

#endif
