/*****************************************************************************\
**                                                                           **
** PBX4Linux                                                                 **
**                                                                           **
**---------------------------------------------------------------------------**
** Copyright: Andreas Eversberg + Joerg Habenicht                            **
**                                                                           **
** rules objects                                                             **
** containes the objects responsible for rules:                              **
** rule_sets, conditions, actions, parameters                                **
**                                                                           **
\*****************************************************************************/ 

#include "rule.h"

#include "main.h"
#include "extension.h"
#include "message.h"
#include "apppbx.h"	// error: this should be in route.h
#include "route.h"

#include <ctype.h>
#include <string.h>
#include <assert.h>


#define PARSE_LINE_SIZE	512


/* easy to comment out debug output */
//#define debug_parse( fmt ... )		printf( fmt  )
#define debug_parse( fmt ... )		((void)0)



/* ---------- debug output -------------*/


/* returns the new position of the cursor "pos" if there
 * are tabulators '\t' in str.
 * can add a start position, if writing of str
 * is not started on position 0 on tty
 *
 * i.e. 
 * "1234", 3 returns 3
 * "\t\t\t4", 3 returns 24
 * "123\t\t\t4", 4 returns 8 */
int get_tabbed_position(const char *str, int pos)
{
  int ret = 0;
  for ( int i=0; i<pos; i++ )
  {
    switch ( str[i] )
    {
    case '\t':
      /* got a tab. move forward "ret" to tabbed position
       * and go on */
      ret /= 8;
      ret ++;	// add one tabulator
      ret *= 8;	// expand to tabbed view
      break;
    case '\r':
    case '\n':
      /* start of line or next line:
       * reset ret to 0 and start over */
      ret = 0;
      break;
    default:
      ret++;
    }
  }
  return ret;
}


/* writes the "string" to tty with a pointer below 
 * to the indexed text.
 * i.e. "		text"
 *      "		^---" */
void markup_output(const char *string, int indexp)
{
  assert(string);
  assert(512 > strlen(string));
  
  char *cpy = strcpy(new char[strlen(string)+1], string);
  char *cindex = strchr(cpy, '\n');
  if (cindex)
    *cindex = '\0';
  
  printf("%s\n", cpy);
  
  for( cindex = cpy; cindex < cpy+indexp; cindex++)
  {
    if ( !isspace(*cindex) )
    {
      *cindex = ' ';
    }
  }
  
  if (*cindex)
    *cindex++ = '^';
  if (*cindex)
    *cindex++ = '-';
  if (*cindex)
    *cindex++ = '-';
  *cindex = '\0';
  
  printf("%s\n", cpy);

  delete[] cpy;  
}

/* service function, returns number of tabs and newlines in the string "src"
 * this way, one can program like:
 *  char *formatstr = spacefmt(new char[strlen(src)+spacecnt(src)+1],src); */
int spacecnt(const char *src)
{
  int ret = 0;
  while (1)
  {
    switch(*src++)
    {
    case '\0':
      return ret;
    case '\t':
    case '\n':
    case '"':
    case '\\':
      ret++;
    default:
      /* nothing */;
    }
  }
  /* not needed, function returns from loop */
  return ret;
}

/* reformats src to target,
 * makes tabs and newlines visible in printout
 *  */
char *spacefmt(char *target, const char *src)
{
  char c, *ret = target;
  assert(ret);
  while (1)
  {
    switch((c=*src++))
    {
    case '\0':
      *target = '\0';
      return ret;
    case '\\':
      *target++ = '\\';
      *target++ = '\\';
      break;
    case '\t':
      *target++ = '\\';
      *target++ = 't';
      break;
    case '\n':
      *target++ = '\\';
      *target++ = 'n';
      break;
    case '"':
      *target++ = '\\';
      *target++ = '"';
      break;
    default:
      *target++ = c;
    }
  }
  /* not needed, function returns from loop */
  *target = '\0';
  return ret;
}

#if 0
/* prints a formatted output with own extension:
 * %ks: print whitespace like coded, i.e. '\t'=>"\\t", '\n'=>"\\n"
 *
 * NOTE: would be "nice to have" to write a string with whitespace formatter
 *       just by writing
 *        printf("%ks",string);
 *       instead of 
 *        char *str = spacefmt(new char[strlen(src)+spacecnt(src)+1],src);
 *        printf("%s",str);
 *        delete str;
 *       but it seems not worth the trouble.
 */
void myprintf(char *format, ...)
{
  char fmstr[128];
  char *fmptr = fmstr;
  char is_format = 0;
  char c;
  va_list ap;
  
  va_start(ap, format);
  while (*format)
  {
    switch ((c=*format++))
    {
    case '%':
      is_format = !is_format;
      fmptr = fmstr;
      goto default_char;
    case 'f':
    case 'i':
    case 'd':
    case 'x':
      if ( is_format )
      {
        *fmptr++ = c;
        *fmptr = '\0';
        switch (c)
        {
        case 'x':
          printf(fmptr, va_arg(ap, int));
          break;
        }
        //printf(fmptr, 
      }
    default:
default_char:    
      if ( !is_format )
      {
        printf("%c", c);
      }
      else
      {
        *fmptr++ = c;
      }
    }
  }
  //vprintf(format, ap);
  va_end(ap);
}
#endif

/*
 Doing error messages like:

----------
While parsing /usr/local/pbx/routing.conf, an error occurred in line 8:
-> h323     : goto ruleset=voip timeout=20 # next line next action
                                           ^
Expecting parameter name.
----------
While parsing /usr/local/pbx/routing.conf, an error occurred in line 9:
->        goto ruleset=extern
              ^
Unknown condition item name 'goto'.
----------
*/                                           
void print_parse_error(const char *fname, const char *line, int lineno, int cursorpos, const char *failure)
{
  printf("While parsing %s, an error occurred in line %d:\n", fname, lineno);
  //printf("-> %s\n", line);
  //memset(pointer, ' ', sizeof(pointer));
  //pointer[p-buffer] = '^';
  //pointer[p-buffer+1] = '\0';
  //printf("   %s\n", pointer);
  markup_output(line, get_tabbed_position(line, cursorpos));
  printf("%s\n", failure);
  //SPRINT(ruleset_error, "Error in file %s, line %d: %s",  filename[nesting], line[nesting], failure);
}


/* ---------- Parser -------------*/


bool Parser::is_allowed_condition_tokenend(char c)
{
  return ( isspace(c) || !c || ':' == c );
}

bool Parser::is_allowed_action_tokenend(char c)
{
  return ( isspace(c) || !c );
}



int Parser::parse_string_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector, bool is_range_possible)
{
    const char *linep;
    const char *valuetok;
    bool is_range_mode = false;
    //StringValue *sv = NULL;
    StringRangeValue *srv = NULL;
    
    assert(line);
    assert(tokenstart);
    assert(tokenlen);
    assert(parsestart >= 0);
    assert(parsestart < (int)strlen(line));
    
    linep = line + parsestart;
    *tokenstart = parsestart;
    
    if ( '=' != *linep )
    {
      goto parse_error;
    }
    linep++;
    (*tokenstart)++;
    
    /* a string token is regular expression: [a-zA-Z0-9][a-zA-Z0-9,_-]*
     * and may be enclosed in quotation marks
     * quotation marks may enclose whitespace, i.e \"[a-zA-Z0-9][a-zA-Z0-9\ \t,_-]*\" */
    if ( '"'==*linep )
    {
      /* TODO: we dont catch escaped quotation marks \"\" here */
#if 0
      linep++;
      while ( *linep && '"' != *linep )
        linep++;
      if ( '"'==*linep )
	linep++;
      else
	/* no matching quotation marks found. error */
	goto parse_error;
#else
      linep++;
      valuetok = linep;
      /*if ( !(isalnum(*linep) && !isspace(*linep)) ) // TODO: we need to allow whitespace here
	goto parse_error;
      linep++;*/
      // TODO: update this to reflect the regular expression above 
      while ( isalnum(*linep) || ' ' == *linep || '\t' == *linep || ',' == *linep || '-' == *linep || '_' == *linep || '/' == *linep )
      {
        if ( '-' == *linep && is_range_possible )
        {
          if ( is_range_mode || 0 >= (linep - valuetok) )
            goto parse_error;
          /* inside range mode */
          is_range_mode = true;
          srv = new StringRangeValue();
          srv->setvalue(valuetok, (linep - valuetok));
          
          valuetok = linep;
          valuetok++;
        }
        else if ( ',' == *linep )
        {
          if ( 0 >= linep - valuetok )
            goto parse_error;
          if ( is_range_mode )
          {
            srv->setvalueto(valuetok, (linep - valuetok));
            valuevector.add(srv);
            
            srv = NULL;
            is_range_mode = false;
          }
          else
          {
            StringValue *sv = new StringValue();
            sv->setvalue(valuetok, (linep - valuetok));
            valuevector.add(sv);
            sv = NULL;
          }
          valuetok = linep;
          valuetok++;
        }
	linep++;
      }
      if ( '"'==*linep )
      {
        if ( 0 >= linep - valuetok )
          goto parse_error;
        if ( is_range_mode )
        {
          srv->setvalueto(valuetok, (linep - valuetok));
          valuevector.add(srv);
          
          srv = NULL;
          is_range_mode = false;
        }
        else
        {
          StringValue *sv = new StringValue();
          sv->setvalue(valuetok, (linep - valuetok));
          valuevector.add(sv);
          sv = NULL;
        }
        //valuetok = linep;
        //valuetok++;
	linep++;
      }
      else
	/* no matching quotation marks found. error */
	goto parse_error;
	
      *tokenlen = linep - (line + *tokenstart);
      goto parse_end;
#endif
    }
    else
    {
      valuetok = linep;
      /*if ( !(isalnum(*linep) && !isspace(*linep)) )
	goto parse_error;
      linep++;*/
      // TODO: update this to reflect the regular expression above 
      while ( (isalnum(*linep) && !isspace(*linep)) || ',' == *linep || '-' == *linep || '_' == *linep || '/' == *linep )
      {
        if ( '-' == *linep && is_range_possible )
        {
          if ( is_range_mode || 0 >= (linep - valuetok) )
            goto parse_error;
          /* inside range mode */
          is_range_mode = true;
          srv = new StringRangeValue();
          srv->setvalue(valuetok, (linep - valuetok));
          
          valuetok = linep;
          valuetok++;
        }
        else if ( ',' == *linep )
        {
          if ( 0 >= linep - valuetok )
            goto parse_error;
          if ( is_range_mode )
          {
            srv->setvalueto(valuetok, (linep - valuetok));
            valuevector.add(srv);
            
            srv = NULL;
            is_range_mode = false;
          }
          else
          {
            StringValue *sv = new StringValue();
            sv->setvalue(valuetok, (linep - valuetok));
            valuevector.add(sv);
            sv = NULL;
          }
          valuetok = linep;
          valuetok++;
        }
	linep++;
      }
    }
    if ( !is_allowed_condition_tokenend(*linep) )
    {
      /* additional token found beond allowed token. error */
      goto parse_error;
    }          	  	  
    
//parse_end:
    if ( 0 >= linep - valuetok )
      goto parse_error;
    if ( is_range_mode )
    {
      srv->setvalueto(valuetok, (linep - valuetok));
      valuevector.add(srv);
      
      srv = NULL;
      is_range_mode = false;
    }
    else
    {
      StringValue *sv = new StringValue();
      sv->setvalue(valuetok, (linep - valuetok));
      valuevector.add(sv);
      sv = NULL;
    }

    *tokenlen = linep - (line + *tokenstart);
parse_end:
    //delete sv;
    delete srv;
    return 0;
    
parse_error:
    *tokenlen = linep - (line + *tokenstart);
    //delete sv;
    delete srv;
    return -1;
}

int Parser::parse_int_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector)
{
    const char *linep;
    const char *valuetok;
    
    int value_from, value_to;
    bool is_range_mode = false;
    
    assert(line);
    assert(tokenstart);
    assert(tokenlen);
    assert(parsestart >= 0);
    assert(parsestart < (int)strlen(line));
    
    linep = line + parsestart;
    *tokenstart = parsestart;
    
    if ( '=' != *linep )
    {
      goto parse_error;
    }
    linep++;
    (*tokenstart)++;
    
    /* an integer token is regular expression: [0-9]+(\-[0-9]+)?(\,[0-9]+(\-[0-9]+)?)*
     * and may be enclosed in quotation marks */
    if ( '"'==*linep )
    {
      /* TODO: we dont catch escaped quotation marks \"\" here */
      linep++;
      // TODO: update this to reflect the regular expression above 
      while ( *linep && '"' != *linep && (isdigit(*linep) || ',' == *linep || '-' == *linep) )
        linep++;
      if ( '"'==*linep )
	linep++;
      else
	/* no matching quotation marks found. error */
	goto parse_error;
    }
    else
    {
      valuetok = linep;
      if ( !isdigit(*linep) )
	goto parse_error;
      linep++;
      // TODO: update this to reflect the regular expression above 
      while ( isdigit(*linep) || ',' == *linep || '-' == *linep )
      {
        if ( '-' == *linep )
        {
          if ( is_range_mode || 0 >= linep - valuetok )
            goto parse_error;
          /* inside range mode */
          is_range_mode = true;
          errno = 0;
          value_from = strtol( valuetok, (char**)&valuetok, 10 );
          if ( errno || (valuetok != linep) )
            goto parse_error;
          valuetok++; // no need to set valuetok = linep anymore
        }
        else if ( ',' == *linep )
        {
          if ( 0 >= linep - valuetok )
            goto parse_error;
          if ( is_range_mode )
          {
            errno = 0;
            value_to = strtol( valuetok, (char**)&valuetok, 10 );
            if ( errno || (valuetok != linep) )
              goto parse_error;
            IntegerRangeValue *iv = new IntegerRangeValue();
            iv->value_to = value_to;
            iv->value = value_from;
            valuevector.add(iv);
            is_range_mode = false;
          }
          else
          {
            errno = 0;
            value_from = strtol( valuetok, (char**)&valuetok, 10 );
            if ( errno || (valuetok != linep) )
              goto parse_error;
            IntegerValue *iv = new IntegerValue();
            iv->value = value_from;
            valuevector.add(iv);
          }
          valuetok++; // no need to set valuetok = linep anymore
        }
	linep++;
      }
    }
    if ( !is_allowed_condition_tokenend(*linep) )
    {
      /* additional token found beond allowed token. error */
      goto parse_error;
    }
    
//parse_end:
    if ( is_range_mode )
    {
      errno = 0;
      value_to = strtol( valuetok, (char**)&valuetok, 10 );
      if ( errno || (valuetok != linep) )
        goto parse_error;
      IntegerRangeValue *iv = new IntegerRangeValue();
      iv->value_to = value_to;
      iv->value = value_from;
      valuevector.add(iv);
      //is_range_mode = false;
    }
    else
    {
      errno = 0;
      value_from = strtol( valuetok, (char**)&valuetok, 10 );
      if ( errno || (valuetok != linep) )
        goto parse_error;
      IntegerValue *iv = new IntegerValue();
      iv->value = value_from;
      valuevector.add(iv);
    }
    
    *tokenlen = linep - (line + *tokenstart);
    return 0;
    
parse_error:
    *tokenlen = linep - (line + *tokenstart);
    return -1;
}


int Parser::parse_yesno_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector)
{
    const char *linep;
    int value = 0;
    int arglen;
    
    assert(line);
    assert(tokenstart);
    assert(tokenlen);
    assert(parsestart >= 0);
    assert(parsestart < (int)strlen(line));
    
    linep = line + parsestart;
    *tokenstart = parsestart;
    
    if ( '=' != *linep )
    {
      goto parse_error;
    }
    linep++;
    (*tokenstart)++;
    
    /* an yesno token is regular expression: (yes|no)
     * and may be enclosed in quotation marks */
    if ( '"'==*linep )
    {
      linep++;
      if ( !strncmp(linep, "yes", (arglen=strlen("yes"))) )
        value = 1;
      else if ( !strncmp(linep, "no", (arglen=strlen("no"))) )
	value = 0;
      else
	goto parse_error;
	
      linep += arglen;
      *tokenlen = linep - (line + *tokenstart);
    
      if ( '"'==*linep )
	linep++;
      else
	/* no matching quotation marks found. error */
	goto parse_error;
    }
    else
    {
      if ( !strncmp(linep, "yes", (arglen=strlen("yes"))) )
	value = 1;
      else if ( !strncmp(linep, "no", (arglen=strlen("no"))) )
	value = 0;
      else
	goto parse_error;
	
      linep += arglen;
      *tokenlen = linep - (line + *tokenstart);
    }
    if ( !is_allowed_condition_tokenend(*linep) )
    {
      /* additional token found beond allowed token. error */
      goto parse_error;
    }
    
//parse_end:
    {
      IntegerValue *iv = new IntegerValue();
      iv->value = value;
      valuevector.add(iv);
    }
    return 0;
    
parse_error:
    *tokenlen = linep - (line + *tokenstart);
    return -1;
}


int Parser::parse_service_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector)
{
    const char *linep;
    int arglen;
    int value = 0;
    
    assert(line);
    assert(tokenstart);
    assert(tokenlen);
    assert(parsestart >= 0);
    assert(parsestart < (int)strlen(line));
    
    linep = line + parsestart;
    *tokenstart = parsestart;
    
    if ( '=' != *linep )
    {
      goto parse_error;
    }
    linep++;
    (*tokenstart)++;
    
    /* an service token is regular expression:
     * (speech|audio|video|digital-restricted|digital-unrestricted|digital-unrestricted-tones)
     * and may be enclosed in quotation marks */
    if ( '"'==*linep )
    {
      linep++;
      if ( !strncasecmp( linep, "speech", (arglen = strlen("speech")) ) )
      {
        value = INFO_CAPABILITY_SPEECH;
      }
      else if ( !strncasecmp( linep, "audio", (arglen = strlen("audio")) ) )
      {
        value = INFO_CAPABILITY_AUDIO;
      }
      else if ( !strncasecmp( linep, "video", (arglen = strlen("video")) ) )
      {
        value = INFO_CAPABILITY_VIDEO;
      }
      else if ( !strncasecmp( linep, "digital-unrestricted-tones", (arglen = strlen("digital-unrestricted-tones")) ) )
      {
        value = INFO_CAPABILITY_DIGITAL_UNRESTRICTED_TONES;
      }
      else if ( !strncasecmp( linep, "digital-unrestricted", (arglen = strlen("digital-unrestricted")) ) )
      {
        value = INFO_CAPABILITY_DIGITAL_UNRESTRICTED;
      }
      else if ( !strncasecmp( linep, "digital-restricted", (arglen = strlen("digital-restricted")) ) )
      {
        value = INFO_CAPABILITY_DIGITAL_RESTRICTED;
      }
      else
	goto parse_error;

      linep += arglen;
      *tokenlen = linep - (line + *tokenstart);
      
      if ( '"'==*linep )
	linep++;
      else
	/* no matching quotation marks found. error */
	goto parse_error;
    }
    else
    {
      if ( !strncasecmp( linep, "speech", (arglen = strlen("speech")) ) )
      {
        value = INFO_CAPABILITY_SPEECH;
      }
      else if ( !strncasecmp( linep, "audio", (arglen = strlen("audio")) ) )
      {
        value = INFO_CAPABILITY_AUDIO;
      }
      else if ( !strncasecmp( linep, "video", (arglen = strlen("video")) ) )
      {
        value = INFO_CAPABILITY_VIDEO;
      }
      else if ( !strncasecmp( linep, "digital-unrestricted-tones", (arglen = strlen("digital-unrestricted-tones")) ) )
      {
        value = INFO_CAPABILITY_DIGITAL_UNRESTRICTED_TONES;
      }
      else if ( !strncasecmp( linep, "digital-unrestricted", (arglen = strlen("digital-unrestricted")) ) )
      {
        value = INFO_CAPABILITY_DIGITAL_UNRESTRICTED;
      }
      else if ( !strncasecmp( linep, "digital-restricted", (arglen = strlen("digital-restricted")) ) )
      {
        value = INFO_CAPABILITY_DIGITAL_RESTRICTED;
      }
      else
	goto parse_error;
	
      linep += arglen;
      *tokenlen = linep - (line + *tokenstart);

    }
    if ( !is_allowed_condition_tokenend(*linep) )
    {
      /* additional token found beond allowed token. error */
      goto parse_error;
    }
    
//parse_end:
    {
      IntegerValue *iv = new IntegerValue();
      iv->value = value;
      valuevector.add(iv);
    }
    return 0;
    
parse_error:
    *tokenlen = linep - (line + *tokenstart);
    return -1;
}


int Parser::parse_diversion_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector)
{
    const char *linep;
    int arglen = 3;
    int value = 0;
    
    assert(line);
    assert(tokenstart);
    assert(tokenlen);
    assert(parsestart >= 0);
    assert(parsestart < (int)strlen(line));
    
    linep = line + parsestart;
    *tokenstart = parsestart;
    
    if ( '=' != *linep )
    {
      goto parse_error;
    }
    linep++;
    (*tokenstart)++;
    
    /* an service token is regular expression: (cfu|cfnr|cfb|cfp)
     * and may be enclosed in quotation marks */
    if ( '"'==*linep )
    {
      arglen = strlen("cfu"); // not needed here, was set ==3 above
      linep++;
      if ( !strncasecmp( linep, "cfu", arglen ) )
        value = INFO_DIVERSION_CFU;
      else if ( !strncasecmp( linep, "cfb", arglen ) )
        value = INFO_DIVERSION_CFB;
      else if ( !strncasecmp( linep, "cfp", arglen ) )
        value = INFO_DIVERSION_CFP;
      else if ( !strncasecmp( linep, "cfnr", (arglen = strlen("cfnr")) ) )
        value = INFO_DIVERSION_CFNR;
      else
	goto parse_error;
      
      linep += arglen;
      *tokenlen = linep - (line + *tokenstart);

      if ( '"'==*linep )
	linep++;
      else
	/* no matching quotation marks found. error */
	goto parse_error;
    }
    else
    {
      arglen = strlen("cfu"); // not needed here, was set ==3 above
      if ( !strncasecmp( linep, "cfu", arglen ) )
        value = INFO_DIVERSION_CFU;
      else if ( !strncasecmp( linep, "cfb", arglen ) )
        value = INFO_DIVERSION_CFB;
      else if ( !strncasecmp( linep, "cfp", arglen ) )
        value = INFO_DIVERSION_CFP;
      else if ( !strncasecmp( linep, "cfnr", (arglen = strlen("cfnr")) ) )
        value = INFO_DIVERSION_CFNR;
      else
	goto parse_error;
      
      linep += arglen;
      *tokenlen = linep - (line + *tokenstart);
    }
    if ( !is_allowed_condition_tokenend(*linep) )
    {
      /* additional token found beond allowed token. error */
      goto parse_error;
    }
    
//parse_end:
    {
      IntegerValue *iv = new IntegerValue();
      iv->value = value;
      valuevector.add(iv);
    }
    return 0;
    
parse_error:
    *tokenlen = linep - (line + *tokenstart);
    return -1;
}


int Parser::parse_conntype_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector)
{  
  const char *linep;
  int arglen;
  int value = 0;
  
  assert(line);
  assert(tokenstart);
  assert(tokenlen);
  assert(parsestart >= 0);
  assert(parsestart < (int)strlen(line));
    
  linep = line + parsestart;
  *tokenstart = parsestart;
  
  if ( '=' != *linep )
  {
    goto parse_error;
  }
  linep++;
  (*tokenstart)++;
  
  /* an yesno token is regular expression: (unknown|subscriber|national|international)
     * and may be enclosed in quotation marks */
  if ( '"'==*linep )
  {
    linep++;
    if ( !strncasecmp( linep, "unknown", (arglen = strlen("unknown")) ) )
      value = INFO_NTYPE_UNKNOWN;
    else if ( !strncasecmp( linep, "subscriber", (arglen = strlen("subscriber")) ) )
      value = INFO_NTYPE_SUBSCRIBER;
    else if ( !strncasecmp( linep, "national", (arglen = strlen("national")) ) )
      value = INFO_NTYPE_NATIONAL;
    else if ( !strncasecmp( linep, "international", (arglen = strlen("international")) ) )
      value = INFO_NTYPE_INTERNATIONAL;
    else
      goto parse_error;

    linep += arglen;
    *tokenlen = linep - (line + *tokenstart);
    
    if ( '"'==*linep )
      linep++;
    else
      /* no matching quotation marks found. error */
      goto parse_error;
  }
  else
  {
    if ( !strncasecmp( linep, "unknown", (arglen = strlen("unknown")) ) )
      value = INFO_NTYPE_UNKNOWN;
    else if ( !strncasecmp( linep, "subscriber", (arglen = strlen("subscriber")) ) )
      value = INFO_NTYPE_SUBSCRIBER;
    else if ( !strncasecmp( linep, "national", (arglen = strlen("national")) ) )
      value = INFO_NTYPE_NATIONAL;
    else if ( !strncasecmp( linep, "international", (arglen = strlen("international")) ) )
      value = INFO_NTYPE_INTERNATIONAL;
    else
      goto parse_error;

    linep += arglen;
    *tokenlen = linep - (line + *tokenstart);
  }
  if ( !is_allowed_condition_tokenend(*linep) )
  {
    /* additional token found beond allowed token. error */
    goto parse_error;
  }
  
//parse_end:
  {
    IntegerValue *iv = new IntegerValue();
    iv->value = value;
    valuevector.add(iv);
  }
  return 0;
  
parse_error:
  *tokenlen = linep - (line + *tokenstart);
  return -1;
}


int Parser::parse_destination_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector)
{
  const char *linep;
  int retval = parse_diversion_argument(line, parsestart, tokenstart, tokenlen, valuevector);
  
  /* in diversion_argument all parameters are already checked, we dont need to assert here anymore */
  if ( retval )
  {
    int arglen;
    
    /* parse missed. now we only have a chance to look for "vbox" or error out */
    linep = line + parsestart;
    *tokenstart = parsestart;
    if ( '=' != *linep )
    {
      goto parse_error;
    }
    linep++;
    (*tokenstart)++;
    if ( '"'==*linep )
    {
      linep++;
      (*tokenstart)++;
      /* previous parse failed, so the token is not cf*
       * now it could only be vbox or an error. */
      if ( strncasecmp( linep, "vbox", (arglen = strlen("vbox")) ) )
	goto parse_error;
      
      linep += arglen;
      *tokenlen = linep - (line + *tokenstart);
      
      if ( '"'==*linep )
	linep++;
      else
	/* no matching quotation marks found. error */
	goto parse_error;
    }
    else
    {
      if ( strncasecmp( linep, "vbox", (arglen = strlen("vbox")) ) )
	goto parse_error;

      linep += arglen;
      *tokenlen = linep - (line + *tokenstart);
    }
    
  }
  else
  {
    /* parse ok. filter out "cfp" (cfu,cfnr,cfb only) and return */
    if ( !strncmp(line+*tokenstart, "cfp", strlen("cfp")) )
      goto parse_error;

    // tokenlen is already set by string parse function
    //linep += arglen;
    //*tokenlen = linep - (line + *tokenstart);
  }
  
  {
    StringValue *sv = new StringValue();
    sv->setvalue(line+*tokenstart, *tokenlen);    
    valuevector.reset();
    valuevector.add(sv);
  }
  return 0;
  
parse_error:
  *tokenlen = linep - (line + *tokenstart);
  return -1;
}


int Parser::parse_time_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector)
{
  int retval = parse_int_argument(line, parsestart, tokenstart, tokenlen, valuevector);
  /* if integer test went ok, we can examine the time arguments */
  if (retval)
    return -1;

  for ( int i=0; i<valuevector.size(); i++ )
  {
    int time;
    const Value *v = valuevector[i];
    switch (v->get_classification())
    {
    case VALUE_TYPE_INTEGER_RANGE:
      time = ((const IntegerRangeValue *)v)->value_to;
      if (time < 0)
        return -1;
      if (time > 2359)
        return -1;
      if (time %100 > 59)
        return -1;
      /* no break */
    case VALUE_TYPE_INTEGER:
      time = ((const IntegerValue *)v)->value;
      if (time < 0)
        return -1;
      if (time > 2359)
        return -1;
      if (time %100 > 59)
        return -1;
      break;
    default:
      /* wanted to have integers instead of other values.
       * error out */
      assert(false);
    }
  }
  return 0;
}


int Parser::parse_mday_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector)
{
  int retval = parse_int_argument(line, parsestart, tokenstart, tokenlen, valuevector);
  /* if integer test went ok, we can examine the time arguments */
  if (retval)
    return -1;

#if 1
  for ( int i=0; i<valuevector.size(); i++ )
  {
    int mday;
    const Value *v = valuevector[i];
    switch (v->get_classification())
    {
    case VALUE_TYPE_INTEGER_RANGE:
      mday = ((const IntegerRangeValue *)v)->value_to;
      if (mday < 0)
        return -1;
      if (mday > 31)
        return -1;
      /* no break */
    case VALUE_TYPE_INTEGER:
      mday = ((const IntegerValue *)v)->value;
      if (mday < 0)
        return -1;
      if (mday > 31)
        return -1;
      /* ok, string may fit into mm with
       * 00 <= mday <= 31 */
      break;
    default:
      /* wanted to have integers instead of other values.
       * error out */
      assert(false);
    }
  }
#else
  const char *endp = line+*tokenstart;
  const char *tokenend = endp+*tokenlen;
  while ( *endp && (endp < tokenend) )
  {
    const char *linep = endp;
    long int mday = strtol(linep, (char **)&endp, 10);
    if (linep == endp)	/* didnt parse anything. error */
      return -1;
    if (mday < 0)
      return -1;
    if (mday > 31)
      return -1;
    /* ok, string may fit into mm with
     * 00 <= mday <= 31 */
    if ( ! *endp )
      break;
    /* maybe this is unnecessary, depends on integer parser */
    if ( ',' == *endp || '-' == *endp )
      endp++;
  }
#endif
  return 0;
}

int Parser::parse_month_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector)
{
  int retval = parse_int_argument(line, parsestart, tokenstart, tokenlen, valuevector);
  /* if integer test went ok, we can examine the time arguments */
  if (retval)
    return -1;

#if 1
  for ( int i=0; i<valuevector.size(); i++ )
  {
    int month;
    const Value *v = valuevector[i];
    switch (v->get_classification())
    {
    case VALUE_TYPE_INTEGER_RANGE:
      month = ((const IntegerRangeValue *)v)->value_to;
      if (month < 1)
        return -1;
      if (month > 12)
        return -1;
      /* no break */
    case VALUE_TYPE_INTEGER:
      month = ((const IntegerValue *)v)->value;
      if (month < 1)
        return -1;
      if (month > 12)
        return -1;
      /* ok, string may fit into mm with
       * 01 <= month <= 12 */
      break;
    default:
      /* wanted to have integers instead of other values.
       * error out */
      assert(false);
    }
  }
#else
  const char *endp = line+*tokenstart;
  const char *tokenend = endp+*tokenlen;
  while ( *endp && (endp < tokenend) )
  {
    const char *linep = endp;
    long int month = strtol(linep, (char **)&endp, 10);
    if (linep == endp)	/* didnt parse anything. error */
      return -1;
    if (month < 1)
      return -1;
    if (month > 12)
      return -1;
    /* ok, string may fit into mm with
     * 01 <= month <= 12 */
    if ( ! *endp )
      break;
    /* maybe this is unnecessary, depends on integer parser */
    if ( ',' == *endp || '-' == *endp )
      endp++;
  }
#endif
  return 0;
}

int Parser::parse_year_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector)
{
  int retval = parse_int_argument(line, parsestart, tokenstart, tokenlen, valuevector);
  /* if integer test went ok, we can examine the time arguments */
  if (retval)
    return -1;

#if 1
  for ( int i=0; i<valuevector.size(); i++ )
  {
    int year;
    const Value *v = valuevector[i];
    switch (v->get_classification())
    {
    case VALUE_TYPE_INTEGER_RANGE:
      year = ((const IntegerRangeValue *)v)->value_to;
      if (year < 1970)
        return -1;
      if (year > 2106)
        return -1;
      /* no break */
    case VALUE_TYPE_INTEGER:
      year = ((const IntegerValue *)v)->value;
      if (year < 1970)
        return -1;
      if (year > 2106)
        return -1;
      /* ok, string may fit into mm with
       * 1970 <= year <= 2106 */
      break;
    default:
      /* wanted to have integers instead of other values.
       * error out */
      assert(false);
    }
  }
#else
  const char *endp = line+*tokenstart;
  const char *tokenend = endp+*tokenlen;
  while ( *endp && (endp < tokenend) )
  {
    const char *linep = endp;
    long int year = strtol(linep, (char **)&endp, 10);
    if (linep == endp)	/* didnt parse anything. error */
      return -1;
    if (year < 1970)
      return -1;
    if (year > 2106)
      return -1;
    /* ok, string may fit into mm with
     * 1970 <= year <= 2106 */
    if ( ! *endp )
      break;
    /* maybe this is unnecessary, depends on integer parser */
    if ( ',' == *endp || '-' == *endp )
      endp++;
  }
#endif
  return 0;
}

int Parser::parse_wday_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector)
{
  int retval = parse_int_argument(line, parsestart, tokenstart, tokenlen, valuevector);
  /* if integer test went ok, we can examine the time arguments */
  if (retval)
    return -1;

#if 1
  for ( int i=0; i<valuevector.size(); i++ )
  {
    int wday;
    const Value *v = valuevector[i];
    switch (v->get_classification())
    {
    case VALUE_TYPE_INTEGER_RANGE:
      wday = ((const IntegerRangeValue *)v)->value_to;
      if (wday < 1)
        return -1;
      if (wday > 7)
        return -1;
      /* no break */
    case VALUE_TYPE_INTEGER:
      wday = ((const IntegerValue *)v)->value;
      if (wday < 1)
        return -1;
      if (wday > 7)
        return -1;
      /* ok, string may fit into mm with
       * 1 <= wday <= 7 */
      break;
    default:
      /* wanted to have integers instead of other values.
       * error out */
      assert(false);
    }
  }
#else
  const char *endp = line+*tokenstart;
  const char *tokenend = endp+*tokenlen;
  while ( *endp && (endp < tokenend) )
  {
    const char *linep = endp;
    long int wday = strtol(linep, (char **)&endp, 10);
    if (linep == endp)	/* didnt parse anything. error */
      return -1;
    if (wday < 1)
      return -1;
    if (wday > 7)
      return -1;
    /* ok, string may fit into mm with
     * 1 <= wday <= 7 */
    if ( ! *endp )
      break;
    /* maybe this is unnecessary, depends on integer parser */
    if ( ',' == *endp || '-' == *endp )
      endp++;
  }
#endif
  return 0;
}


int Parser::parse_interface_argument(const char *line, int parsestart, int *tokenstart, int *tokenlen, ValueVector &valuevector)
{
  /* <interface>:<channel> */
  const char *linep;
  
  assert(line);
  assert(tokenstart);
  assert(tokenlen);
  assert(parsestart >= 0);
  assert(parsestart < (int)strlen(line));
  
  linep = line + parsestart;
  *tokenstart = parsestart;
  
  if ( '=' != *linep )
  {
    goto parse_error;
  }
  linep++;
  (*tokenstart)++;
  
  /* a string token is regular expression: [a-zA-Z0-9]+\:[a-zA-Z0-9]+
   * and may be enclosed in quotation marks */
  if ( '"'==*linep )
  {
    assert(false); // FIXME
    /* TODO: we dont catch escaped quotation marks \"\" here */
    linep++;
    (*tokenstart)++;
    while ( *linep && '"' != *linep )
      linep++;

    *tokenlen = linep - (line + *tokenstart);
    if ( '"'==*linep )
      linep++;
    else
      /* no matching quotation marks found. error */
      goto parse_error;
  }
  else
  {
    if ( !(isalnum(*linep) && !isspace(*linep)) )
      goto parse_error;
    linep++;
    while ( (isalnum(*linep) && !isspace(*linep)) )
      linep++;
    if ( ':' != *linep )
      goto parse_error;
    linep++;
    if ( !(isalnum(*linep) && !isspace(*linep)) )
      goto parse_error;
    linep++;
    while ( (isalnum(*linep) && !isspace(*linep)) )
      linep++;

    *tokenlen = linep - (line + *tokenstart);
  }
  if ( !is_allowed_condition_tokenend(*linep) )
  {
    /* additional token found beond allowed token. error */
    goto parse_error;
  }          	  	  
  
//parse_end:
  {
    StringValue *sv = new StringValue();
    sv->setvalue(line+*tokenstart, *tokenlen);    
    valuevector.add(sv);
  }
  return 0;
    
parse_error:
  *tokenlen = linep - (line + *tokenstart);
  return -1;
}



/* ---------- Value -------------*/

Value::Value(int clas)
 : classification(clas)
{
}

Value::Value()
 : classification(VALUE_TYPE_NULL)
{
}

int Value::get_classification() const
{
  return classification;
}

struct route_param *Value::fill_route_param(struct route_param *fill) const
{
  assert(fill);
  fill->value_type = classification;
  
  return fill;
}


struct route_cond *Value::fill_route_cond(struct route_cond *fill) const
{
  assert(fill);
  fill->value_type = classification;
  
  return fill;
}


IntegerValue::IntegerValue(int clas)
 : Value(clas), value(0)
{
}

IntegerValue::IntegerValue()
 : Value(VALUE_TYPE_INTEGER), value(0)
{
}

struct route_param *IntegerValue::fill_route_param(struct route_param *fill) const
{
  Value::fill_route_param(fill); // assert inclusive
  fill->integer_value = value;
    
  return fill;
}

struct route_cond *IntegerValue::fill_route_cond(struct route_cond *fill) const
{
  Value::fill_route_cond(fill); // assert inclusive
  fill->integer_value = value;
    
  return fill;
}

IntegerRangeValue::IntegerRangeValue()
 : IntegerValue(VALUE_TYPE_INTEGER_RANGE), value_to(0)
{
}

/* I think we can delete this method.
 * There are no ranges in action parameter arguments.
 * Should we better assert() here? */
struct route_param *IntegerRangeValue::fill_route_param(struct route_param *fill) const
{
  IntegerValue::fill_route_param(fill); // assert inclusive
  
  return fill;
}

struct route_cond *IntegerRangeValue::fill_route_cond(struct route_cond *fill) const
{
  IntegerValue::fill_route_cond(fill); // assert inclusive
  fill->integer_value_to = value_to;
    
  return fill;
}


StringValue::StringValue(int clas)
 : Value(clas), value(NULL)
{
}

StringValue::StringValue()
 : Value(VALUE_TYPE_STRING), value(NULL), autodelete_value(false)
{
}

StringValue::~StringValue()
{
  if ( autodelete_value )
  {
    delete[] value;
  }
}


void StringValue::setvalue(const char *str, int len)
{
  if ( autodelete_value )
    delete[] value;
  autodelete_value = true;

  if ( len < 0 )
  {
    value = strcpy( new char[ strlen(str) +1 ], str );
  }
  else
  {
    len = strnlen(str, len);
    value = strncpy( new char[len+1], str, len );
    value[len] = '\0';
  }
}


int StringValue::parse(const char *str)
{
  assert(false);
  /* looks for existence of ',' or '\0' or whitespace and
   * copies content upto there.
   * includes whitespace, if string enclosed in '"'
   * includes ',' if escaped like "\\," */
  const char *linep;
  bool inside_paren = false;
  
  assert(str);
  
  while ( isspace(*str) )
    str++;
  linep = str;
  if ( '"' == *linep )
  {
    inside_paren = true;
    linep++;
  }

  return linep - str;
}

struct route_param *StringValue::fill_route_param(struct route_param *fill) const
{
  Value::fill_route_param(fill); // assert inclusive
  fill->string_value = strcpy( (char *)malloc(strlen(value)+1), value );
  rmemuse++; // Andreas' memory usage trace
  
  return fill;
}

struct route_cond *StringValue::fill_route_cond(struct route_cond *fill) const
{
  Value::fill_route_cond(fill); // assert inclusive
  fill->string_value = strcpy( (char *)malloc(strlen(value)+1), value );
  rmemuse++; // Andreas' memory usage trace

  return fill;
}


StringRangeValue::StringRangeValue()
 : StringValue(VALUE_TYPE_STRING_RANGE), value_to(NULL)
{
}

StringRangeValue::~StringRangeValue()
{
  if ( autodelete_value )
  {
    delete[] value_to;
  }
}


void StringRangeValue::setvalueto(const char *str, int len)
{
  if ( autodelete_value )
    delete[] value_to;
  autodelete_value = true;

  if ( len < 0 )
  {
    value_to = strcpy( new char[ strlen(str) +1 ], str );
  }
  else
  {
    len = strnlen(str, len);
    value_to = strncpy( new char[len+1], str, len );
    value_to[len] = '\0';
  }
}

/* I think we can delete this method.
 * There are no ranges in action parameter arguments.
 * Should we better assert() here? */
struct route_param *StringRangeValue::fill_route_param(struct route_param *fill) const
{
  StringValue::fill_route_param(fill); // assert inclusive
  
  return fill;
}

struct route_cond *StringRangeValue::fill_route_cond(struct route_cond *fill) const
{
  StringValue::fill_route_cond(fill); // assert inclusive
  fill->string_value_to = strcpy( (char *)malloc(sizeof(value_to)+1), value_to );
  rmemuse++; // Andreas' memory usage trace
  fill->comp_string = strcmp(value, value_to);
    
  return fill;
}


/* ---------- Condition -------------*/

Condition::Condition()
 :args(NULL)
{
}

Condition::~Condition()
{
  delete[] args;
}



/* a rule consists of "[condition[arguments] [condition[arguments] [..]]] : action [parameter [..]]\n"
 * If the line contains only whitespace or '#' comments, we have no rule and
 * may go on searching in the next line
 *
 * this parser gets its token from *line and parses up to
 * + a recognised token,
 * + a recognised ruleset definition,
 * + the ':' char,
 * + the end of the line (or comment),
 * + the end of the file.
 * 
 * With "parsepos" the position of the parser in the line is returned. This way
 * the next call does not have to parse the old token again, and may get a
 * reduced line.
 *
 * i.e parses CONDITION1=CARG1 CONDITION2
 *   CONDITION1=CARG1 CONDITION2 : action param1=parg1 param2=parg2
 *
 * return 0 if successful
 * return -1 if parse error
 * return 1 if EOF
 * return 2 if new ruleset starts ("[ruleset]")
 * return 3 if empty line (only whitespace or comment)
 * return 4 if no more conditions found (but ':') */
int Condition::parse(const char *line, const char *fname, int lineno, int *parsepos)
{
  const char *linep;

  int arglen = 0;
  int tokenstart = 0, tokenlen = 0;
  
  int c;
  int retval = 0;
  
  assert(line);
  assert(fname);
  assert(parsepos);
  assert(*parsepos >= 0);
  assert(*parsepos < (int)strlen(line));

  /* start decoding at the position of the last deconding process */
  linep = line + *parsepos;
  
  while (isspace((c = *linep++)))
    if ('\n' == c || '\r' == c)
    {
      *parsepos = linep - line -1;
      return 3;
    }
    
  if ( ':' == c )
  {
    /* end of conditional part. now check for actions */
    *parsepos = linep - line -1;
    return 4;
  }

  if ( '\0' == c || '#' == c )
  {
    /* end of line buffer or end of conditional part.
     * you may check the end of file and continue */
    *parsepos = linep - line -1;
    return 3;
  }
  

  if ( '[' == c )
  {
    /* got a ruleset definition.
     * allowed operation, if no other condition was decoded.
     * (i.e. "condition1 [ruleset2] : action")
     * let the rule object decide what to do */
    linep--;
    *parsepos = linep - line;
    retval = 2;
    goto condition_parse_end;
  }    

  /* ok, this is no whitespace and no ruleset definition.
   * there has to be some meaning here */

  linep--; // now we do point to a ruleset or a condition definition
  
  /* try to find the end of the token, so we have a better match against
   * the token list. e.g. avoid matching "timeout" against "time" */
  arglen = 0;
  while ( linep[arglen] && isalnum(linep[arglen]) && !isspace(linep[arglen]) && '=' != linep[arglen] && ':' != linep[arglen] )
    arglen++;

  /* try to find a corresponding token from the condition list.
   * case is not important.
   * linear search is ok */
  struct cond_defs *pcon;
  int cindex;
  for ( pcon = cond_defs, cindex = 0; (pcon->name); pcon++, cindex++ )
  {
    if ( !strncasecmp(pcon->name, linep, arglen) )
      break;
  }
  index = cindex;
  
  if ( !pcon->name )
  {
    /* token not found, error exit */
    goto condition_parse_error;
  }


  match = pcon->match;
  type = pcon->type;
  linep += strlen(pcon->name);
  *parsepos = linep - line;
  debug_parse("condition: found token \"%s\", test args at pos. %d.. ",
         pcon->name, *parsepos);

  
  /* crude test for correct parameter.
   * regular expression matching would be better, and
   * could be placed inside the struct cond_defs definition */
  
  switch (pcon->type)
  {
  case COND_TYPE_NULL:
    /* no args, check for next ':' */
    //debug_parse("no args\n");
    goto condition_case_done;
    //break;
  case COND_TYPE_INTEGER:
    retval = parse_int_argument(line, *parsepos, &tokenstart, &tokenlen, vv);
    break;
  case COND_TYPE_TIME:
    retval = parse_time_argument(line, *parsepos, &tokenstart, &tokenlen, vv);
    break;
  case COND_TYPE_MDAY:
    retval = parse_mday_argument(line, *parsepos, &tokenstart, &tokenlen, vv);
    break;
  case COND_TYPE_MONTH:
    retval = parse_month_argument(line, *parsepos, &tokenstart, &tokenlen, vv);
    break;
  case COND_TYPE_WDAY:
    retval = parse_wday_argument(line, *parsepos, &tokenstart, &tokenlen, vv);
    break;
  case COND_TYPE_YEAR:
    retval = parse_year_argument(line, *parsepos, &tokenstart, &tokenlen, vv);
    break;
  case COND_TYPE_STRING:
    retval = parse_string_argument(line, *parsepos, &tokenstart, &tokenlen, vv);
    break;
  case COND_TYPE_IP:
    /* currently not used. update when needed */
    printf(" TODO: parse matching condition type!\n");
    assert(false);
    break;
  case COND_TYPE_SERVICE:
    retval = parse_service_argument(line, *parsepos, &tokenstart, &tokenlen, vv);
    break;
  case COND_TYPE_IFATTR:
    retval = parse_interface_argument(line, *parsepos, &tokenstart, &tokenlen, vv);
    break;
  default:
    printf("internal error: no matching condition type found for token %s. type=%d\n", pcon->name, pcon->type);
    goto condition_parse_error;
  }

  if (retval < 0)
  {
    *parsepos += tokenlen;
    goto condition_parse_error;
  }
  args = strncpy( new char[tokenlen +1], line+tokenstart, tokenlen );
  args[tokenlen] = '\0';
  *parsepos = tokenstart + tokenlen;
  linep = line + *parsepos;

condition_case_done:
  debug_parse("found args \"%s\"\n", args);
  
  /* verify that there are no additional characters after the token
   * only whitespace and ':' are allowed */
  if ( !(isspace(*linep) || ':'==*linep) )
  {
    printf("error: additional information found for token %s\n", pcon->name);
    goto condition_parse_error;
  }

  /* condition name found, we can stop here and
   * continue parsing actions or more conditions */
  *parsepos = linep - line;
  goto condition_parse_end;
  
condition_parse_error:
  *parsepos = linep - line;
  printf("Failure in \"%s\", line %d, position %d.\n", fname, lineno, get_tabbed_position(line,*parsepos)+1);
  markup_output(line, *parsepos);
  //printf("containing \"%s\"\n", line);
  retval = -1;
  
condition_parse_end:  
  debug_parse("condition.parse: returning %d\n", retval);
  return retval;
}


int Condition::debug_out(FILE *outf) const
{
  int ret = 0;
  assert(outf);
  
  /* search for name of condition */
  const char *name = NULL;
#if 1
  name = cond_defs[index].name;
#else
  for (struct cond_defs *con = cond_defs; con->name; con++)
  {
    if ( id == con->match )
    {
      name = con->name;
      break;
    }
  }
#endif
  
  assert(name);
  if ( args )
    ret = fprintf(outf, "%s=%s ", name, args);
  else
    ret = fprintf(outf, "%s ", name);

  return ret;
}


#if 0
struct route_cond { /* an item */
	struct route_cond	*next;			/* next entry */
	int 			index;			/* index of cond_defs */
	int			match;			/* what is matching (MATCH_*) */
	int			value_type;		/* type of value (VALUE_TYPE_*) */
	int			value_extension;	/* will it be extended? */
	int			integer_value;		/* first integer */
	int			integer_value_to;	/* second integer */
	char			*string_value;		/* first string */
	char			*string_value_to;	/* second string */
	int			comp_string;		/* compare value of strings */
};
#endif
struct route_cond *Condition::get_struct() const
{
  struct route_cond *ret = (struct route_cond *)calloc(1, sizeof(struct route_cond));
  if ( !ret )
    return ret;
  rmemuse++; // Andreas' memory usage trace
  
  ret->index = index;
  ret->match = match;
  if ( vv.size() > 0 )
  {
#if 1
    vv[0]->fill_route_cond(ret);
    struct route_cond **rc = &ret;
    for (int i=1; i<vv.size(); i++)
    {
      (*rc)->value_extension = 1;
      rc = &(*rc)->next;
      *rc = (struct route_cond *)calloc(1, sizeof(struct route_cond));
      if ( !*rc )
        break;
      rmemuse++; // Andreas' memory usage trace
      (*rc)->index = index;
      (*rc)->match = match;
      vv[i]->fill_route_cond(*rc);
    }
#else
    Value *v = vv.get()[0];

    /* this switch-case could be solved with rtti,
     * but I hesitate to enable it for this project */
    switch ( v->get_classification() )
    {
    case VALUE_TYPE_STRING_RANGE:
      StringRangeValue *srv = (StringRangeValue *)v;
      ret->comp_string = strcmp(srv->value, srv->value_to);
      ret->string_value_to = strcpy( (char *)malloc( strlen(srv->value_to) +1 ), srv->value_to );
      rmemuse++; // Andreas' memory usage trace
      /* no break */
    case VALUE_TYPE_STRING:
      StringValue *sv = (StringValue *)v;
      ret->string_value = strcpy( (char *)malloc( strlen(sv->value) +1 ), sv->value );
      rmemuse++; // Andreas' memory usage trace
      break;
    case VALUE_TYPE_INTEGER_RANGE:
      IntegerRangeValue *irv = (IntegerRangeValue *)v;
      ret->integer_value_to = irv->value_to;
      /* no break */
    case VALUE_TYPE_INTEGER:
      IntegerValue *iv = (IntegerValue *)v;
      ret->integer_value = iv->value;
      break;
    default:
      ;
    }
#endif

  }
  // TODO !!! andere Felder nachtragen
  return ret;
}



/* ---------- ActionParameter -------------*/


ActionParameter::ActionParameter()
// : id(PARAM_TYPE_NULL), args(NULL)
{
  args = NULL;
}

ActionParameter::~ActionParameter()
{
  delete[] args;
}

const char *ActionParameter::get_txt() const
{
#if 1
  return param_defs[route_param_index].name;
#else
  struct param_defs *pdef;
  for ( pdef = param_defs; (pdef->name); pdef++ )
  {
    if ( pdef->id == id )
      return pdef->name;
  }
  /* nothing found? can't be!! */
  assert(false);
#endif
}

const char *ActionParameter::get_arg() const
{
  return args;
}


int ActionParameter::debug_out(FILE *outf) const
{
  int ret = 0;
  assert(outf);
  
  /* search for name of action */
  // TODO: use route_param_index
  const char *name = NULL;
#if 1
  name = param_defs[route_param_index].name;
#else
  for (struct param_defs *pd = param_defs; pd->name; pd++)
  {
    if ( id == pd->id )
    {
      name = pd->name;
      break;
    }
  }
#endif
  
  assert(name);
  if ( args )
    ret = fprintf(outf, "%s=%s ", name, args);
  else
    ret = fprintf(outf, "%s ", name);

  return ret;
}


struct route_param *ActionParameter::get_struct() const
{
  /* we need to feed information into
   * "index", "id", "integer_value", "string_value" */
  struct route_param *ret = (struct route_param *)calloc(1, sizeof(struct route_param));
  if ( !ret )
    return ret;
  rmemuse++; // Andreas' memory usage trace
  
  ret->index = route_param_index;
  ret->id = id;

  if ( vv.size() > 0 )
  {
    vv[0]->fill_route_param(ret);
    struct route_param **rp = &ret;
    for (int i=1; i<vv.size(); i++)
    {
      (*rp)->value_extension = 1;
      rp = &(*rp)->next;
      *rp = (struct route_param *)calloc(1, sizeof(struct route_param));
      if ( !*rp )
        break;
      rmemuse++; // Andreas' memory usage trace
      (*rp)->index = route_param_index;
      (*rp)->id = id;
      vv[i]->fill_route_param(*rp);
    }
  }
  else
  {
    ret->value_type = VALUE_TYPE_NULL;
  }  
  
  /*if ( args )
  {
    ret->string_value = strcpy( (char *)malloc(strlen(args)+1), args );
    rmemuse++; // Andreas' memory usage trace
    ret->integer_value = atoi( args );
  }*/
  
  return ret;
}


/* ---------- Action -------------*/

Action::Action()
 : id(0), action_defs_index(0), is_parsetest(false)
{
}

/* this routine parses the action part of the line.
 * looks for the first occurence of ':' in the line and 
 * decodes the following tokens
 * i.e parses ACTION and following
 *   condition1=carg1 condition2 : ACTION param1=parg1 param2=parg2
 *				   ACTION2 param1b param2b
 *
 * return 0 if successful
 * return -1 if parse error (including whitespace or comment only)
 * return 1 if EOF */
int Action::parse(const char *line, const char *fname, int lineno, int *parsepos)
{
  const char *linep;
  int tokenstart = 0, tokenlen = 0;
  int c;
  int retval = 0;
  ValueVector xvv;
  

  assert(line);
  assert(fname);
  assert(parsepos);
  assert(*parsepos >= 0);
  assert(*parsepos < (int)strlen(line));

  linep = line + *parsepos;
  
  
  while (isspace((c = *linep++)))
    if ('\n' == c || '\r' == c)
    {
      goto action_parse_error;
    }
    
  if ( '\0' == c || '#' == c )
  {
    /* end of line buffer. check end of file and continue */
    goto action_parse_error;
  }

  if ( '[' == c )
  {
    /* got a ruleset definition. forbidden operation. */
    goto action_parse_error;
  }    

  
  /* ok, this is no whitespace and no ruleset definition.
   * there has to be some meaning here */

  linep--; // now we do point to a ruleset or a condition definition

  //debug_parse("action: testing string \"%s\"\n", linep );

  /* try to find the end of the token, so we have a better match against
   * the token list. e.g. avoid matching "timeout" against "time"
   * the token has the reg exp. [a-zA-Z0-9_-]+ */
  int arglen;
  arglen = 0;
  while ( linep[arglen] && 
          (isalnum(linep[arglen]) || '-' == linep[arglen] || '_' == linep[arglen]) && 
          !isspace(linep[arglen]) && 
          '=' != linep[arglen] && 
          ':' != linep[arglen] )
    arglen++;

  /* try to find a corresponding token from the condition list.
   * case is not important.
   * linear search is ok */
  struct action_defs *paction;
  int aindex;
  for ( paction = action_defs, aindex = 0; (paction->name); paction++, aindex++ )
  {
    if ( ! strncasecmp(paction->name, linep, arglen) )
      break;
  }

  if ( !paction->name )
  {
    /* token not found, error exit */
    goto action_parse_error;
  }

  id = paction->id;
  action_defs_index = aindex;
  linep += strlen(paction->name);
  *parsepos = linep - line;
  debug_parse("action: verified token \"%s\", testing args at position %d.. ",
         paction->name, *parsepos);

  if ( !(*linep) )
  {
    debug_parse("\n");
    goto action_parse_end;
  }


  /* verify that there are no additional characters after the token
   * only whitespace is allowed */
  if ( !isspace(*linep) )
  {
    debug_parse("\nerror: additional information found for token %s\n", paction->name);
    goto action_parse_error;
  }
  linep++;

  while ( *linep )
  {
    while ( isspace(*linep) )
      linep++;

    if ( !*linep || '#' == *linep ) // reached comment or end of line
    {
      goto action_parse_end;
    }
    
    /* no space and not at end of string, there might be a parameter */
    /* try to find the end of the token, so we have a better match against
    * the token list. e.g. avoid matching "timeout" against "time"
    * the token has the reg exp. [a-zA-Z0-9_-]+ */
    //int arglen;
    arglen = 0;
    while ( linep[arglen] && 
            (isalnum(linep[arglen]) || '-' == linep[arglen] || '_' == linep[arglen]) && 
            !isspace(linep[arglen]) && 
            '=' != linep[arglen] && 
            ':' != linep[arglen] )
      arglen++;
  
    struct param_defs *pdef;
    int route_param_index;
    for ( pdef = param_defs, route_param_index=0; (pdef->name); pdef++, route_param_index++ )
    {
      if ( ! strncasecmp(pdef->name, linep, arglen) )
        break;
    }

    if ( !pdef->name )
    {
      /* token not found, error exit */
      debug_parse("\n");
      goto action_parse_error;
    }
    linep += strlen(pdef->name);
    *parsepos = linep - line;

    char *args = NULL;
    xvv.reset();
    //int arglen = 0;
    switch (pdef->type)
    {
    case PARAM_TYPE_NULL:
      /* nothing to do, continue */
      goto action_case_done;
    case PARAM_TYPE_INTEGER:
      retval = parse_int_argument(line, *parsepos, &tokenstart, &tokenlen, xvv);
      break;
    case PARAM_TYPE_STRING:
      retval = parse_string_argument(line, *parsepos, &tokenstart, &tokenlen, xvv);
      break;
    case PARAM_TYPE_YESNO:
      retval = parse_yesno_argument(line, *parsepos, &tokenstart, &tokenlen, xvv);
      break;
    case PARAM_TYPE_SERVICE:
      /* speech|audio|video|digital-restricted|digital-unrestricted|digital-unrestricted-tones */
      retval = parse_service_argument(line, *parsepos, &tokenstart, &tokenlen, xvv);
      break;
    case PARAM_TYPE_DIVERSION:
      retval = parse_diversion_argument(line, *parsepos, &tokenstart, &tokenlen, xvv);
      break;
    case PARAM_TYPE_DESTIN:
      retval = parse_destination_argument(line, *parsepos, &tokenstart, &tokenlen, xvv);
      break;
    case PARAM_TYPE_PORTS:
      /* currently not used. please implement when needed */
      assert(false);
      break;
    case PARAM_TYPE_TYPE:
      /* no break */
    case PARAM_TYPE_CALLERIDTYPE:
      /* unknown|subscriber|national|international */
      retval = parse_conntype_argument(line, *parsepos, &tokenstart, &tokenlen, xvv);
      break;
    default:
      debug_parse("\naction: internal error: parameter type %d not recognised\nexiting\n", pdef->type);
    }    

    *parsepos = tokenstart + tokenlen;
    if (retval < 0)
    {
      goto action_parse_error;
    }
    linep = line + *parsepos;
    args = strncpy( new char[tokenlen +1], line+tokenstart, tokenlen );
    args[tokenlen] = '\0';

action_case_done:
    
    /* verify that there are no additional characters after the token
     * only whitespace is allowed */
    if ( !isspace(*linep) )
    {
      debug_parse("\naction: error: additional information found for token %s\n", pdef->name);
      delete args;
      goto action_parse_error;
    }
    linep++;

    ActionParameter *ap = new ActionParameter();
    apv.add(ap);
    ap->route_param_index = route_param_index;
    ap->id = pdef->id;
    ap->args = args;
    ap->vv = xvv;
    
    xvv.set_all_autodelete(false); // dont delete object of this array, they are referenced elsewhere
    debug_parse("found \"%s\" arg \"%s\"\n", ap->get_txt(), ap->get_arg() /*?ap->get_arg():"(none)"*/ );

  }

  goto action_parse_end;
    
action_parse_error:
  if ( !is_parsetest )
  {
    printf("Failure in \"%s\", line %d, position %d.\n", fname, lineno, get_tabbed_position(line,*parsepos)+1);
    markup_output(line, *parsepos);
    //printf("containing \"%s\"\n", line);
  }
  retval = -1;

action_parse_end:
  return retval;
}

int Action::debug_out(FILE *outf) const
{
  int ret = 0;

  /* search for name of action */
  const char *name = NULL;
  for (struct action_defs *ad = action_defs; ad->name; ad++)
  {
    if ( id == ad->id )
    {
      name = ad->name;
      break;
    }
  }
  
  assert(name);
  ret = fprintf(outf, "%s ", name);
  
  /* now print parameters of the action */
  
  for (int i=0; ret>=0 && i<apv.size(); i++)
  {
    assert(apv[i]);
    ret += apv[i]->debug_out(outf);
  }

  ret += fprintf(outf, "\n");
  return ret;
}


struct route_action *Action::get_struct() const
{
  struct route_action *ret = (struct route_action*)calloc(1, sizeof(struct route_action));
  if ( !ret )
  {
    return ret;
  }
  rmemuse++; // Andreas' memory usage trace

  ret->index = action_defs_index;
  
  struct route_param **rp = &ret->param_first;
  for( int i=0; i<apv.size(); i++ )
  {
    assert(apv[i]);
    *rp = apv[i]->get_struct();
    if ( !*rp )
      break;
    /* ActionParameter::get_struct() may deliver more than one struct.
     * additional parameters are attached to (*rp)->next.
     * We need to walk along the linked list, until next==NULL */
    while ( (*rp) )
      rp = &(*rp)->next;
  }
  
  /* search parameters for timeout value */
  for( int i=0; i<apv.size(); i++ )
  {
    if ( apv[i]->id == PARAM_TIMEOUT )
    {
      ret->timeout = atoi(apv[i]->args);
      break;
    }
  }
  
  return ret;
}


/* ---------- Rule -------------*/


/* parses a rule line by line. the rule ends with a newline character.
 * each line is searched for a new ruleset definition ([ruleset]).
 * if so, the parser exits immediately.
 * otherwise each line is given one or more times to the condition object,
 * each recognised condition is removed from the string,
 * so the next condition object only sees a condition or ':'.
 * afterwards the ':' is removed from the string and the result
 * is given to the action object to parse.
 *
 * return 0 if successfull
 * return -1 if parse error (message was already written)
 * return 1 if EOF
 * return 2 if new ruleset token was recognised. filepointer points to beginning of token */
int Rule::parse(FILE *route_conf, const char *fname, int *lineno)
{
  Condition *pcondition = NULL;
  Action *paction = NULL;
  char *line = new char[PARSE_LINE_SIZE];
  char *linep;
  int cond_ret, action_ret;
  int retval = 0;
  int parser_position_in_line = 0;
  
  assert(route_conf);
  assert(fname);
  assert(lineno);

rule_outer_loop:
  if ( !feof(route_conf) )
  {    
    linep = fgets( line, PARSE_LINE_SIZE, route_conf );
    if ( !linep )
    {
      /* error occurred or EOF while not reading anything */
      if ( feof(route_conf) )
      {
        retval = 1;
        goto rule_parse_end;
      }
    }
    (*lineno)++;
    
    /* max line size exceeded? */
    if (strlen(line) >= PARSE_LINE_SIZE-1) 
    {
      printf("Maximum line size of %d bytes exceeded\n", PARSE_LINE_SIZE-1);
      goto rule_parse_error;
    }
    
    {
      char *nstr = spacefmt(new char[strlen(line)+spacecnt(line)+1],line);
      debug_parse("rule: testing string \"%s\"\n", nstr);
      delete[] nstr;
    }
    
rule_iterate_condition:
    pcondition = new Condition();
    cond_ret = pcondition->parse(line, fname, *lineno, &parser_position_in_line);
    switch ( cond_ret )
    {
    case -1:
      retval = -1;
      goto rule_parse_end;
    case 1:
      /* end of line? evaluate again */
      delete pcondition;
      pcondition = NULL;
      goto rule_outer_loop;
    case 2:
      /* Ruleset token found. If this is the first one in line its OK.
       * If there was a condition found previously, its an error.
       * If ok, rewind file for length of line and return from rule parsing */
      if ( cv.size() )
      {
        goto rule_parse_error;
      }
      fseek( route_conf, - strlen(line), SEEK_CUR );
      (*lineno)--;
      retval = 2;
      goto rule_parse_end;
    case 3:
      /* nothing found? try again next line */
      delete pcondition;
      pcondition = NULL;
      goto rule_outer_loop;
    case 4:
      /* no more conditions found, but beginning of action part */
      break;
    case 0:
      /* we found a condition. iterate again, if there is one more condition */
      // TODO: check here if same condition had been previously.
      cv.add(pcondition);
      goto rule_iterate_condition;
    default:
      assert(false);
    }

    if ( ':' != line[parser_position_in_line] )
    {
      debug_parse("error: colon not found.\n");
      goto rule_parse_error;
    }
    parser_position_in_line++; // move beond the ':'
    

    paction = new Action();
    action_ret = paction->parse(line, fname, *lineno, &parser_position_in_line);
    if (action_ret < 0)
    {
      /* some error happened.
       * either EOF occured (ret==1),
       * the next token is no simple rule(==2)
       * or a parse error (==-1) */
      /* error in parsing. immediate abort and return error */
      goto rule_parse_error;
    }

rule_iterate_action:
    av.add(paction);
    paction = NULL;	// if we do exit, dont delete this valid action
    
    /* get the next line. maybe another action
     * is defined for the same condition
     * i.e.: condition : ACTION1
     *                   ACTION2	*/
     
    linep = fgets( line, PARSE_LINE_SIZE, route_conf );
    if ( !linep )
    {
      /* error occurred or EOF while not reading anything */
      if ( feof(route_conf) )
      {
        retval = 1;
        goto rule_parse_end;
      }
    }
    (*lineno)++;
    parser_position_in_line = 0;
    
    /* max line size exceeded? */
    if (strlen(line) >= PARSE_LINE_SIZE-1) 
    {
      printf("Maximum line size of %d bytes exceeded\n", PARSE_LINE_SIZE-1);
      goto rule_parse_error;
    }
    
    paction = new Action();
    paction->is_parsetest = true; // only test if this line is ok, dont call out on error
    action_ret = paction->parse(line, fname, *lineno, &parser_position_in_line);
    if (action_ret < 0)
    {
      /* some error happened.
       * either EOF occured (ret==1),
       * the next token is no simple rule(==2)
       * or a parse error (==-1) */
      /* error in parsing. maybe we have seen a condition.
       * revert the line fetching and return normal */
      fseek( route_conf, - strlen(line), SEEK_CUR );
      (*lineno)--;
      goto rule_parse_end;
    }
    goto rule_iterate_action;    
  }
  else
  {
    retval = 1;
  }

  goto rule_parse_end;
rule_parse_error:

  printf("Failure in \"%s\", line %d.\n", fname, *lineno);
  markup_output(line, parser_position_in_line);
  //printf("containing \"%s\"\n", line);
  retval = -1;

rule_parse_end:
  delete pcondition;
  delete paction;
  delete[] line;
  return retval;
}

int Rule::debug_out(FILE *outf) const
{
  int ret = 0;
  
  for (int i=0; ret>=0 && i<cv.size(); i++)
  {
    assert(cv[i]);
    ret = cv[i]->debug_out(outf);
  }
  if (ret<0)
    return ret;
    
  fprintf(outf, ":");
  
  for (int i=0; ret>=0 && i<av.size(); i++)
  {
    assert(av[i]);
    ret = av[i]->debug_out(outf);
  }
  if (ret<0)
    return ret;

  return ret;
}

struct route_rule *Rule::get_struct() const
{
  struct route_rule *ret = (struct route_rule*)calloc(1, sizeof(struct route_rule));
  if ( !ret )
  {
    return ret;
  }
  rmemuse++; // Andreas' memory usage trace

  {
    struct route_cond **rc = &ret->cond_first;
    for( int i=0; i<cv.size(); i++ )
    {
      assert(cv[i]);
      *rc = cv[i]->get_struct();
      if ( !*rc )
        break;
      rc = &(*rc)->next;
    }
  }


  {
    struct route_action **ra = &ret->action_first;
    for( int i=0; i<av.size(); i++ )
    {
      assert(av[i]);
      *ra = av[i]->get_struct();
      if ( !*ra )
        break;
      ra = &(*ra)->next;
    }
  }


  return ret;
}



/* --------- Ruleset -----------*/

Ruleset::Ruleset()
 : name(NULL)
{
}

Ruleset::~Ruleset()
{
  delete[] name;
}


int Ruleset::parse(FILE *route_conf, const char *fname, int *lineno)
{
  char *line = new char[PARSE_LINE_SIZE];
  //char line[PARSE_LINE_SIZE];
  char *linep, *tokenendp;
  int c;
  
  assert(lineno);
  assert(route_conf);
  assert(fname);
  
ruleset_outer_loop:
  while ( !feof(route_conf) )
  {
    linep = fgets( line, PARSE_LINE_SIZE, route_conf );
    if ( !linep )
    {
      /* error occurred or EOF while not reading anything */
      if ( feof(route_conf) )
      {
        return 1;
      }
    }
    (*lineno)++;
    
    /* max line size exceeded? */
    if (strlen(line) >= PARSE_LINE_SIZE-1) 
    {
      debug_parse("Maximum line size of %d bytes exceeded\n", PARSE_LINE_SIZE-1);
      goto ruleset_parse_error;
    }
    
    while (isspace((c = *linep++)))
      if ('\n' == c || '\r' == c)
      {
        goto ruleset_outer_loop; /* same as continue for outer loop */
      }
      
    if ( '\0' == c || '#' == c )
    {
      /* end of line buffer. check end of file and continue */
      continue;
    }
    
    if ( '[' != c )
    {
      debug_parse("Expecting ruleset name starting with '['.\n");

      goto ruleset_parse_error;
    }
    
    tokenendp = strchr(linep, ']');
    if ( !tokenendp )
    {
      debug_parse("End of token descriptor ']' not found in line.\n");
      
      goto ruleset_parse_error;
    }
    
    *tokenendp = '\0';
    
    name = strcpy(new char[strlen(linep)+1], linep);
    debug_parse("ruleset: found token \"%s\"\n", name);
    
    /* rule set name found, we can stop here and continue parsing rules */
    break;
  }
  delete[] line;
  line = NULL;
  
  if ( feof(route_conf) )
  {
    return 2;
  }
  
  int rule_ret;
  do {
  
    Rule *rule = new Rule();
    rule_ret = rule->parse(route_conf, fname, lineno);
    if ( rule_ret )
    {
      /* some error happened.
       * either EOF occured (ret==1),
       * the next token is no simple rule(==2)
       * or a parse error (==-1) */
      delete rule;
      if (rule_ret < 0)
      {
        /* error in parsing. immediate abort and return error */
        return -1;
      }
      /* up to now parse has been ok. thus return 0.
       * EOF is recognised on the next parse run. */
      break;
    }
    rulevector.add(rule);
    
  } while ( !rule_ret );
  
  return 0;
  
ruleset_parse_error:

  /* prepare the config line to display on tty. snip any \n\r */
  linep = line;
  while ( *linep != '\n' && *linep != '\r' && *linep != '\0' )
    linep++;
  *linep = '\0';
  
  printf("Failure in \"%s\", line %d.\n", fname, *lineno);
  markup_output(line, (linep - line));
  //printf("containing \"%s\"\n", line);
  printf("expected ruleset definition (a.k. \"[ruleset]\")\n");

  /*printf("While parsing %s, an error occurred in line %d:\n", filename[nesting], line[nesting]);
  printf("-> %s\n", buffer);
  memset(pointer, ' ', sizeof(pointer));
  pointer[p-buffer] = '^';
  pointer[p-buffer+1] = '\0';
  printf("   %s\n", pointer);
  printf("%s\n", failure);
  SPRINT(ruleset_error, "Error in file %s, line %d: %s",  filename[nesting], line[nesting], failure);*/
  
  delete[] line;
  return -1;
}

const char * const Ruleset::get_name() const
{
  return name;
}

int Ruleset::debug_out(FILE *outf) const
{
  int ret = 0;
  assert(outf);
  
  ret = fprintf(outf, "[%s]\n", name);
  if (ret < 0)
    return -1;
  ret = 0;
  
  for( int i=0; ret>=0 && i<rulevector.size(); i++ )
  {
    assert(rulevector[i]);
    ret = rulevector[i]->debug_out(outf);
  }
  ret = fprintf(outf, "\n");
  return ret;
}

struct route_ruleset *Ruleset::get_struct() const
{
  struct route_ruleset *ret = (struct route_ruleset*)calloc(1, sizeof(struct route_ruleset));
  if ( !ret )
    return ret;
  rmemuse++; // Andreas' memory usage trace

  /* name entry is hardcoded 64 byte size: char name[64]; */
  strncpy(ret->name, name, 63);
  
  struct route_rule **rr = &ret->rule_first;
  for( int i=0; i<rulevector.size(); i++ )
  {
    assert(rulevector[i]);
    *rr = rulevector[i]->get_struct();
    if ( !*rr )
      break;
    rr = &(*rr)->next;
  }
  
  return ret;
}


RulesetVector RulesetRoot::ruleset;

int RulesetRoot::parse_all(FILE *route_conf, const char *fname)
{
  Ruleset	*rs;
  int		rs_ret;
  int		line_num = 0;

  /* empty the ruleset before filling */
  ruleset.reset();

  do 
  {  
    rs = new Ruleset();
    rs_ret = rs->parse(route_conf, fname, &line_num);
    if (rs_ret)
    {
      if (rs_ret < 0)
      {
        printf("Error parsing %s\nexit\n", fname);
	delete rs;
        return -1;
      }
      
      /* end of parsing (file end?). done now, return */
      delete rs;
      return 0;
    }
    ruleset.add(rs);
  } while ( !rs_ret );
  
  return 0;
}

const RulesetVector &RulesetRoot::get_vector()
{
  return ruleset;
}

int RulesetRoot::debug_out(FILE *outf)
{
  int ret = 0;
  assert(outf);
  
  for( int i=0; ret>=0 && i<ruleset.size(); i++ )
  {
    assert(ruleset[i]);
    ret = ruleset[i]->debug_out(outf);
  }
  
  return ret;
}

struct route_ruleset *RulesetRoot::get_struct()
{
  if ( ruleset.size() <= 0 )
    return NULL;

  struct route_ruleset *ret = ruleset[0]->get_struct();
  if ( !ret )
    return ret;

  struct route_ruleset **rs = &ret->next;
  for( int i=1; i<ruleset.size(); i++ )
  {
    assert(ruleset[i]);
    *rs = ruleset[i]->get_struct();
    if ( !*rs )
      break;
    rs = &(*rs)->next;
  }
  
  return ret;
}



#if 1
int test_read_config(const char *fname)
{
#if 0
  char src[] = "this tab\t,this newline\n";
  char *formatstr = spacefmt(new char[strlen(src)+spacecnt(src)+1],src);

  debug_parse("\"%s\"\"%s\"\n",src, formatstr);
  delete[] formatstr;
  return 0;
#endif

  FILE *f;
  
  assert(fname);
  f = fopen(fname, "r");
  if ( !f )
  {
    printf("Could not open %s\n", fname);
    return -1;
  }
  
  RulesetRoot::parse_all( f, fname );
  
  fclose(f);
  
  /* print out parsed info */
  //RulesetRoot::debug_out(stdout);
  
  //delete rsv;
  return 0;
}
#endif
