/*****************************************************************************\
**                                                                           **
** PBX4Linux                                                                 **
**                                                                           **
**---------------------------------------------------------------------------**
** Copyright: Andreas Eversberg + Joerg Habenicht                            **
**                                                                           **
** vector header file                                                        **
** contains interface for generic arrays of objects (just like STL)          **
**                                                                           **
\*****************************************************************************/ 

#ifndef VECTOR_H
#define VECTOR_H

#include <assert.h>


template <class ArrayItem> class Vector
{
  
protected:
  ArrayItem *array;
  int arraysize;
  int itemnum;
  
  static const int startsize;
  static const int incrsize ;
  void copy(const Vector &copy); // used from Vector() and '=' operator
  
public:
  Vector();
  Vector(const Vector &copy);
  ~Vector();
  
  void add(const ArrayItem item);	// add a pointer to end of list
  int remove(const ArrayItem item);	// remove that pointer, return 1 if succeed, 0 otherwise
  void reset();				// empties the vector of all content
  ArrayItem const * const get() const;	// get a list of the pointers.
  int size() const;			// get the size of the list

  void operator=(const Vector &copy);
  ArrayItem operator[](int pos) const;
};





template <class ArrayItem> class AutodeleteVector: public Vector<ArrayItem>
{
protected:
  bool *do_delete_item;
  void copy(const AutodeleteVector &copy); // used from AutodeleteVector() and '=' operator

public:
  AutodeleteVector();
  AutodeleteVector(AutodeleteVector &copy);
  ~AutodeleteVector();
  
  void add(const ArrayItem item);	// add a pointer to end of list
  void add(const ArrayItem item, bool doAutodelete);	// add a pointer to end of list
  int remove(const ArrayItem item);	// remove that pointer, return 1 if succeed, 0 otherwise
  void reset();				// empties the vector of all content
  void set_all_autodelete(bool doAutodelete);	// (un)sets all entries to autodelete
  
  void operator=(const AutodeleteVector &copy);
};


template <class ArrayItem> const int Vector<ArrayItem>::startsize = 10; // 10 elements
template <class ArrayItem> const int Vector<ArrayItem>::incrsize = 10;

template <class ArrayItem> Vector<ArrayItem>::Vector()
 : arraysize(startsize), itemnum(0)
{
  array = new ArrayItem [arraysize];
}

template <class ArrayItem> Vector<ArrayItem>::Vector(const Vector &copy)
{
  copy(copy);
}

template <class ArrayItem> Vector<ArrayItem>::~Vector()
{
  delete[] array;
}


template <class ArrayItem> void Vector<ArrayItem>::copy(const Vector &copy)
{
  arraysize = copy.arraysize;
  delete[] array;
  array = new ArrayItem [arraysize];
  itemnum = copy.itemnum;
  for (int i=0; i<itemnum; i++)
  {
    array[i] = copy.array[i];
  }
}

template <class ArrayItem> void Vector<ArrayItem>::add(const ArrayItem item)
{
  /* if the storage is full, get a new storage and copy all content into
   * then add the item at the end */
  assert (itemnum <= arraysize);
  if ( itemnum >= arraysize )
  {
    ArrayItem *newarray;

    arraysize += incrsize;
    newarray = new ArrayItem [arraysize];
    for (int i=0; i<itemnum; i++)
    {
      newarray[i] = array[i];
    }
    delete[] array;
    array = newarray;
  }
  
  array[itemnum++] = item;
}

template <class ArrayItem> int Vector<ArrayItem>::remove(const ArrayItem item)
{
  int i;
  int have_found = 0;
  
  for (i=0; i<itemnum; i++)
  {
    if ( item == array[i] )
    {
      /* found one. move the rest, closing the gap */
      have_found = 1;
      for ( ; i<(itemnum-1); i++ )
      {
        array[i] = array[i+1];
      }
      
      itemnum--;
      
      break;
    }
  }

  return have_found;
}

template <class ArrayItem> void Vector<ArrayItem>::reset()
{
  itemnum = 0;
}

template <class ArrayItem> ArrayItem const * const Vector<ArrayItem>::get() const
{
  return array;
}

template <class ArrayItem> int Vector<ArrayItem>::size() const
{
  return itemnum;
}

template <class ArrayItem> void Vector<ArrayItem>::operator=(const Vector &copy)
{
  this->copy(copy);
}

template <class ArrayItem> ArrayItem Vector<ArrayItem>::operator[](int pos) const
{
  assert(pos < itemnum);
  return array[pos];
}


template <class ArrayItem> AutodeleteVector<ArrayItem>::AutodeleteVector()
 : Vector<ArrayItem>()
{
  do_delete_item = new bool [Vector<ArrayItem>::arraysize];
}

template <class ArrayItem> AutodeleteVector<ArrayItem>::AutodeleteVector(AutodeleteVector &copy)
{
  this->copy(copy);
}

template <class ArrayItem> AutodeleteVector<ArrayItem>::~AutodeleteVector()
{
  for (int i=0; i<Vector<ArrayItem>::itemnum; i++)
  {
    if ( do_delete_item[i] )
      delete Vector<ArrayItem>::array[i];
  }

  delete[] do_delete_item;
}



template <class ArrayItem> void AutodeleteVector<ArrayItem>::copy(const AutodeleteVector &copy)
{
  Vector<ArrayItem>::copy(copy);
  delete[] do_delete_item;
  do_delete_item = new bool [Vector<ArrayItem>::arraysize];
  for (int i=0; i<Vector<ArrayItem>::itemnum; i++)
  {
    do_delete_item[i] = copy.do_delete_item[i];
  }
}

template <class ArrayItem> void AutodeleteVector<ArrayItem>::add(const ArrayItem item)
{
  this->add(item, true);
}

template <class ArrayItem> void AutodeleteVector<ArrayItem>::add(const ArrayItem item, bool doAutodelete)
{
  /* if the storage is full, get a new storage and copy all content into
   * then add the item at the end */
  int asize = Vector<ArrayItem>::arraysize;
  Vector<ArrayItem>::add(item);
  if ( asize != Vector<ArrayItem>::arraysize )
  {
    bool *newbool = new bool[Vector<ArrayItem>::arraysize];
    
    for (int i=0; i<(Vector<ArrayItem>::itemnum-1); i++)
    {
      newbool[i] = do_delete_item[i];
    }
    delete[] do_delete_item;
    do_delete_item = newbool;
  }
  
  do_delete_item[Vector<ArrayItem>::itemnum-1] = doAutodelete;
}

template <class ArrayItem> int AutodeleteVector<ArrayItem>::remove(const ArrayItem item)
{
  int i;
  int have_found = 0;
  
  for (i=0; i<Vector<ArrayItem>::itemnum; i++)
  {
    if ( item == Vector<ArrayItem>::array[i] )
    {
      /* found one. move the rest, closing the gap */
      have_found = 1;
      for ( ; i<(Vector<ArrayItem>::itemnum-1); i++ )
      {
        Vector<ArrayItem>::array[i] = Vector<ArrayItem>::array[i+1];
        do_delete_item[i] = do_delete_item[i+1];
      }
      
      Vector<ArrayItem>::itemnum--;
      
      break;
    }
  }

  return have_found;
}

template <class ArrayItem> void AutodeleteVector<ArrayItem>::reset()
{
  for (int i=0; i<Vector<ArrayItem>::itemnum; i++)
  {
    if ( do_delete_item[i] )
      delete Vector<ArrayItem>::array[i];
  }

  Vector<ArrayItem>::itemnum = 0;
}

template <class ArrayItem> void AutodeleteVector<ArrayItem>::set_all_autodelete(bool doAutodelete)
{
  for (int i=0; i<Vector<ArrayItem>::itemnum; i++)
  {
    do_delete_item[i] = doAutodelete;
  }
}

template <class ArrayItem> void AutodeleteVector<ArrayItem>::operator=(const AutodeleteVector &copy)
{
  this->copy(copy);
}



#endif
