// Opal PBX media interface


#include <ptlib.h>

#include "opal_pbxms.h"
#include <opal/mediastrm.h>
#include <opal/patch.h>



#define new PNEW


PBXMediaStream::PBXMediaStream(const OpalMediaFormat & mediaFormat, unsigned sessionID, BOOL isSource, unsigned long portid)
  : OpalMediaStream(mediaFormat, sessionID, isSource),
{
	port_id = portid;
	PDEBUG(DEBUG_OPAL, "Media stream created. (port %d)\n", port_id);
}

PBXMediaStream::~PBXMediaStream()
{
	PDEBUG(DEBUG_OPAL, "Media stream terminated. (port %d)\n", port_id);
}
 
 
BOOL PBXMediaStream::Open()
{
	if (isOpen)
		return TRUE;

	BOOL res = OpalMediaStream::Open();

	PDEBUG(DEBUG_OPAL, "Media stream opened with media format %s. (port %d)\n", mediaFormat.GetString(), port_id);
 
	return res;
}
 
BOOL PBXMediaStream::Start()
{
	PDEBUG(DEBUG_OPAL, "Media stream is a %s. (port %d)\n", IsSink() ? "sink" : "source", port_id);

	return OpalMediaStream::Start();
}
  
BOOL PBXMediaStream::Close()
{
	BOOL res = OpalMediaStream::Close();

	PDEBUG(DEBUG_OPAL, "Media stream of media format %s closed. (port %d)\n", mediaFormat.GetString(), port_id);

	return res;
}
 

//
// Requests data from connected mISDN loop channel.
//
BOOL PBXMediaStream::ReadData(BYTE * buffer, PINDEX size, PINDEX & length)
{
	PDEBUG(DEBUG_OPAL, "Read data of %d bytes max", size);
	length = 0;
  
	if (IsSink())
	{
		PTRACE(1, "Media\tTried to read from sink media stream");
		return FALSE;
	}

	if (!isOpen)
	{
		PTRACE(3, "Media\tStream has been closed, so exit now");
		return FALSE;
	}

	// Get audio data from port
	do {
		PBX_LOCK;
		struct OPALPort *port = find_port_id(port_id);
		if (port)
		{
			int rx_read = port->p_rx_read;
			unsigned char *rx_data = port->p_rx_data;
			int rx_len = (port->p_rx_write - rx_read) & RX_DATA_MASK;
			int i = 0;
			length = (rx_len>size) ? size : rx_size;
			while(i < length)
			{
				*buffer++ = rx_data[rx_read];
				rx_read = (rx_read+1) & RX_DATA_MASK;
				i++;
			}
		} else
			length = 0;
		PBX_UNLOCK;

		if (length)
			return TRUE;

		// No audio data
		PThread::Sleep(10);
		if (connection.GetPhase() == OpalConnection::ReleasedPhase)
			break;
	} while (isOpen);

	PDEBUG(DEBUG_OPAL, "Exit now from opal media stream. (port %d)\n", port_id);
	return FALSE;
}


//
// Send data to connected mISDN loop channel.
//
BOOL PBXMediaStream::WriteData(const BYTE * buffer, PINDEX length, PINDEX & written)
{
	if (IsSource()) {
		PTRACE(1, "Media\tTried to write to source media stream");
		return FALSE;
	}

	PBX_LOCK;
	struct OPALPort *port = find_port_id(port_id);
	if (port)
	{
		while(length)
		{
			int len = (length>1024) ? 1024 : length;
			port->tx_function(buffer, len);
			buffer += len;
			length -= len;
		}
	}
	PBX_UNLOCK;

      	written = length;
	return TRUE;
}


//
// We get synchronous data from mISDN channel and may send jittered data.
//
BOOL PBXMediaStream::IsSynchronous() const
{
	if (IsSource())
		return TRUE;

	/**We are reading from mISDN, which generates a frame at a regular rate*/
	return TRUE;
}


