/*****************************************************************************\
**                                                                           **
** PBX4Linux                                                                 **
**                                                                           **
**---------------------------------------------------------------------------**
** Copyright: Andreas Eversberg                                              **
**                                                                           **
** opal port                                                                 **
**                                                                           **
\*****************************************************************************/ 

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include "main.h"


/*
 * initialize opal port
 */
OPALPort::OPALPort(int type, char *portname, struct port_settings *settings) : Port(type, portname, settings)
{
	/* configure device */
	switch (type)
	{
		case PORT_TYPE_OPAL_IN:
		break;
		case PORT_TYPE_OPAL_OUT:
		SPRINT(p_name, "OPAL_outgoing_port_#%lu", p_serial);
		break;
	}
}


/*
 * destructor
 */
OPALPort::~OPALPort()
{
}


/*
 * endpoint sends messages to the interface
 */
int OPALPort::message_epoint(unsigned long epoint_id, int message_id, union parameter *param)
{
	PBXConnection *connection;
	PBXConnection::CallEndReason opal_cause;
	char name[sizeof(p_name)];

	if (Port::message_epoint(epoint_id, message_id, param))
		return(1);

	switch(message_id)
	{
		case MESSAGE_mISDNSIGNAL: /* isdn command */
		case MESSAGE_INFORMATION: /* additional digits from endpoint */
		PDEBUG(DEBUG_OPAL, "OPALPort(%s) mISDN signal not supported.\n", p_name);
		break;

		case MESSAGE_PROCEEDING: /* call of endpoint is proceeding */
		PDEBUG(DEBUG_OPAL, "OPALPort(%s) OPAL port with (caller id %s) received proceeding\n", p_name, p_callerinfo.id);
		if (p_state != PORT_STATE_IN_OVERLAP)
		{
			PERROR("OPALPort(%s) proceeding command only possible in setup state.\n", p_name);
			break;
		}
		p_state = PORT_STATE_IN_PROCEEDING;
		break;

		case MESSAGE_ALERTING: /* call of endpoint is ringing */
		PDEBUG(DEBUG_OPAL, "OPALPort(%s) OPAL port with (caller id %s) received alerting\n", p_name, p_callerinfo.id);
		if (p_state != PORT_STATE_IN_OVERLAP
		 && p_state != PORT_STATE_IN_PROCEEDING)
		{
			PERROR("OPALPort(%s) alerting command only possible in setup or proceeding state.\n", p_name);
			break;
		}
		p_state = PORT_STATE_IN_ALERTING;
		UCPY(name, p_name);
		connection = OPAL_ep->FindConnectionWithLock(name);
		if (connection)
		{
			if (options.OPAL_ringconnect && !p_callerinfo.intern[0])
			{
				connection->AnsweringCall(OPALConnection::AnswerCallNow);
				p_state = PORT_STATE_CONNECT;
			} else
				connection->AnsweringCall(OPALConnection::AnswerCallPending);
			connection->Unlock();
		}
		break;

		case MESSAGE_CONNECT: /* call of endpoint is connected */
		PDEBUG(DEBUG_OPAL, "OPALPort(%s) OPAL port with (caller id %s) received connect\n", p_name, p_callerinfo.id);
		if (p_state != PORT_STATE_IN_OVERLAP
		 && p_state != PORT_STATE_IN_PROCEEDING
		 && p_state != PORT_STATE_IN_ALERTING)
		{
			PDEBUG(DEBUG_OPAL, "OPALPort(%s) connect command only possible in setup, proceeding or alerting state.\n", p_name);
			break;
		}
		new_state(PORT_STATE_CONNECT);
		/* copy connected information */
		memcpy(&p_connectinfo, &param->connectinfo, sizeof(p_connectinfo));
		p_connectinfo.itype = INFO_ITYPE_OPAL;
		UCPY(name, p_name);
		connection = OPAL_ep->FindConnectionWithLock(name);
		if (connection)
		{
#if 0
			int type, present, screen;
			PString connect_number;
			/* modify connectinfo (COLP) */
			if (p_connectinfo.present!=INFO_PRESENT_NULL)
			{
				connect_number = p_connectinfo.id;
				switch(p_connectinfo.ntype)
				{
					case INFO_NTYPE_SUBSCRIBER:
					type = Q931::SubscriberType;
					break;
					case INFO_NTYPE_NATIONAL:
					type = Q931::NationalType;
					break;
					case INFO_NTYPE_INTERNATIONAL:
					type = Q931::InternationalType;
					break;
					default: /* INFO_TYPE_UNKNOWN */
					type = Q931::UnknownType;
				}
				switch(p_connectinfo.present)
				{
					case INFO_PRESENT_RESTRICTED:
					present = 1;
					break;
					case INFO_PRESENT_NOTAVAIL:
					present = 2;
					break;
					default: /* INFO_PRESENT_ALLOWED */
					present = 0;
				}
				switch(p_connectinfo.screen)
				{
					case INFO_SCREEN_USER:
					screen = 0;
					break;
					default: /* INFO_SCREEN_NETWORK */
					screen = 3;
				}
				if (p_OPAL_connect)
				{
//PDEBUG(DEBUG_OPAL, "DDDEBUG: number %s, type=%d, present %d, screen %d\n", p_connectinfo.id, type, present, screen);
					((Q931 *)p_OPAL_connect)->SetConnectedNumber(connect_number, Q931::ISDNPlan, type, present, screen);
				}
				else
					PERROR("missing p_OPAL_connect\n");
			}
#endif
			connection->AnsweringCall(OPALConnection::AnswerCallNow);
			connection->Unlock();
		}
		mutex_OPAL.Wait();
		break;

		case MESSAGE_DISCONNECT: /* call has been disconnected */
#if 0
		PDEBUG(DEBUG_OPAL, "OPALPort(%s) OPAL port with (caller id %s) received disconnect cause=%d\n", p_name, p_callerinfo.id, param->disconnectinfo.cause);
		/* we just play what we hear from the remote site */
		if (p_state == PORT_STATE_IN_OVERLAP
		 || p_state == PORT_STATE_IN_PROCEEDING)
		{
			/* copy connected information */
			memcpy(&p_connectinfo, &param->connectinfo, sizeof(p_connectinfo));
			UCPY(name, p_name);
			connection = OPAL_ep->FindConnectionWithLock(name);
			if (connection)
			{
				connection->AnsweringCall(OPALConnection::AnswerCallNow);
				connection->Unlock();
			}
		}
		new_state(PORT_STATE_DISCONNECT);
		break;
#endif

		case MESSAGE_RELEASE: /* release OPAL port */
		PDEBUG(DEBUG_OPAL, "OPALPort(%s) OPAL port with (caller id %s) received disconnect cause=%d\n", p_name, p_callerinfo.id, param->disconnectinfo.cause);
		if (p_state != PORT_STATE_IN_OVERLAP
		 && p_state != PORT_STATE_IN_PROCEEDING
		 && p_state != PORT_STATE_IN_ALERTING
		 && p_state != PORT_STATE_OUT_SETUP
		 && p_state != PORT_STATE_OUT_OVERLAP
		 && p_state != PORT_STATE_OUT_PROCEEDING
		 && p_state != PORT_STATE_OUT_ALERTING
		 && p_state != PORT_STATE_CONNECT)
		{
			PERROR("OPALPort(%s) disconnect command only possible in setup, proceeding, alerting or connect state.\n", p_name);
			break;
		}

		switch(param->disconnectinfo.cause)
		{
			case 1:
			OPAL_cause = OPALConnection::EndedByNoUser;
			break;

			case 2:
			case 3:
			case 5:
			OPAL_cause = OPALConnection::EndedByUnreachable;
			break;

			case 17:
			OPAL_cause = OPALConnection::EndedByRemoteBusy;
			break;
	
			case 18:
			OPAL_cause = OPALConnection::EndedByNoEndPoint;
			break;
	
			case 19:
			OPAL_cause = OPALConnection::EndedByNoAnswer;
			break;
	
			case 21:
			OPAL_cause = OPALConnection::EndedByRefusal;
			break;
	
			case 27:
			OPAL_cause = OPALConnection::EndedByHostOffline;
			break;
	
			case 47:
			OPAL_cause = OPALConnection::EndedByConnectFail;
			break;
	
			case 65:
			OPAL_cause = OPALConnection::EndedByCapabilityExchange;
			break;
	
			case 42:
			OPAL_cause = OPALConnection::EndedByRemoteCongestion;
			break;
	
			case 41:
			OPAL_cause = OPALConnection::EndedByTemporaryFailure;
			break;
	
			default:
			OPAL_cause = OPALConnection::EndedByRemoteUser;
			break;
		}
		UCPY(name, p_name);
		OPAL_ep->ClearCall(name, OPAL_cause);

		delete this;
		break;

		case MESSAGE_SETUP: /* dial-out command received from epoint */
		PDEBUG(DEBUG_OPAL, "OPALPort(%s) OPAL port received setup from '%s' to '%s'\n", p_name, param->setup.callerinfo.id, param->setup.dialinginfo.number);
		if (p_type!=PORT_TYPE_OPAL_OUT)
		{
			PERROR("OPALPort(%s) cannot dial because OPAL port not of outgoing type.\n", p_name);
			break;
		}
		if (p_state != PORT_STATE_IDLE)
		{
			PERROR("OPALPort(%s) error: dialing command only possible in idle state.\n", p_name);
			break;
		}

		/* link relation */
		if (p_epointlist)
		{
			PERROR("OPALPort(%s) software error: epoint pointer is set in idle state, how bad!! exitting.\n", p_name);
			exit(-1);
		}
		if (!(epointlist_new(epoint_id)))
		{
			PERROR("no memory for epointlist\n");
			exit(-1);
		}

		/* copy setup infos to port */
		memcpy(&p_callerinfo, &param->setup.callerinfo, sizeof(p_callerinfo));
		memcpy(&p_dialinginfo, &param->setup.dialinginfo, sizeof(p_dialinginfo));
		memcpy(&p_bearerinfo, &param->setup.bearerinfo, sizeof(p_bearerinfo));
		memcpy(&p_redirinfo, &param->setup.redirinfo, sizeof(p_redirinfo));

		p_state = PORT_STATE_OUT_SETUP;

		UCPY(name, p_name);
		OPAL_ep->Call(name, param->setup.callerinfo.id, param->setup.dialinginfo.number);
		break;

		default:
		PDEBUG(DEBUG_OPAL, "OPALPort(%s) OPAL port with (caller id %s) received an unsupported message: %d\n", p_name, p_callerinfo.id, message_id);
	}

	return(1);
}


