/*****************************************************************************\
**                                                                           **
** PBX4Linux                                                                 **
**                                                                           **
**---------------------------------------------------------------------------**
** Copyright: Andreas Eversberg                                              **
**                                                                           **
** message types and parameters                                              **
**                                                                           **
\*****************************************************************************/ 

#ifndef MESSAGE_H
#define MESSAGE_H

enum { /* interface types */
	INFO_ITYPE_EXTERN,
	INFO_ITYPE_INTERN,
	INFO_ITYPE_H323,
	INFO_ITYPE_SIP,
	INFO_ITYPE_VBOX,
};

enum { /* number types */
	INFO_NTYPE_UNKNOWN,
	INFO_NTYPE_SUBSCRIBER,
	INFO_NTYPE_NATIONAL,
	INFO_NTYPE_INTERNATIONAL,
};

enum { /* number presentation */
	INFO_PRESENT_NULL,
	INFO_PRESENT_ALLOWED,
	INFO_PRESENT_RESTRICTED,
	INFO_PRESENT_NOTAVAIL,
};

enum { /* number presentation */
	INFO_SCREEN_USER, /* user provided */
	INFO_SCREEN_NETWORK, /* network provided */
};

enum { /* redirection reason */
	INFO_REDIR_UNKNOWN,
	INFO_REDIR_BUSY,
	INFO_REDIR_NORESPONSE,
	INFO_REDIR_UNCONDITIONAL,
	INFO_REDIR_OUTOFORDER,
	INFO_REDIR_CALLDEFLECT,
};

#define	INFO_NOTIFY_NONE			0x00
#define	INFO_NOTIFY_USER_SUSPENDED		0x80	
#define	INFO_NOTIFY_USER_RESUMED		0x81
#define	INFO_NOTIFY_BEARER_SERVICE_CHANGED	0x82
#define	INFO_NOTIFY_CALL_COMPLETION_DELAY	0x83
#define	INFO_NOTIFY_CONFERENCE_ESTABLISHED	0xc2
#define	INFO_NOTIFY_CONFERENCE_DISCONNECTED	0xc3
#define INFO_NOTIFY_OTHER_PARTY_ADDED		0xc4
#define INFO_NOTIFY_ISOLATED			0xc5
#define INFO_NOTIFY_REATTACHED			0xc6
#define INFO_NOTIFY_OTHER_PARTY_ISOLATED	0xc7
#define INFO_NOTIFY_OTHER_PARTY_REATTACHED	0xc8
#define INFO_NOTIFY_OTHER_PARTY_SPLIT		0xc9
#define INFO_NOTIFY_OTHER_PARTY_DISCONNECTED	0xca
#define INFO_NOTIFY_CONFERENCE_FLOATING		0xcb
#define INFO_NOTIFY_CONFERENCE_DISCONNECTED_P	0xcc /* preemted */
#define INFO_NOTIFY_CONFERENCE_FLOATING_S_U_P	0xcf /* served user preemted */
#define INFO_NOTIFY_CALL_IS_A_WAITING_CALL	0xe0
#define INFO_NOTIFY_DIVERSION_ACTIVATED		0xe8
#define INFO_NOTIFY_RESERVED_CT_1		0xe9
#define INFO_NOTIFY_RESERVED_CT_2		0xea
#define INFO_NOTIFY_REVERSE_CHARGING		0xee
#define INFO_NOTIFY_REMOTE_HOLD			0xf9
#define INFO_NOTIFY_REMOTE_RETRIEVAL		0xfa
#define INFO_NOTIFY_CALL_IS_DIVERTING		0xfb

enum { /* diversion types */
	INFO_DIVERSION_CFU,
	INFO_DIVERSION_CFNR,
	INFO_DIVERSION_CFB,
	INFO_DIVERSION_CFP,
};

/* bearer capability */
#define	INFO_CAPABILITY_SPEECH				0x10
#define	INFO_CAPABILITY_AUDIO				0x11
#define	INFO_CAPABILITY_VIDEO				0x22
#define	INFO_CAPABILITY_DIGITAL_UNRESTRICTED		0x23
#define	INFO_CAPABILITY_DIGITAL_RESTRICTED		0x24
#define	INFO_CAPABILITY_DIGITAL_UNRESTRICTED_TONES	0x35
#define CLASS_CAPABILITY_AUDIO				0x10
#define CLASS_CAPABILITY_DATA				0x20

enum { /* bearer mode */
	INFO_MODE_CIRCUIT,
	INFO_MODE_PACKET,
};

enum { /* user l1 */
	INFO_CODEC_DATA,
	INFO_CODEC_ULAW,
	INFO_CODEC_ALAW,
	INFO_CODEC_G721,
	INFO_CODEC_H221H242,
	INFO_CODEC_V110,
	INFO_CODEC_NONCCITT,
	INFO_CODEC_V120,
	INFO_CODEC_X31HDLC,
};

enum { /* isdnsignal */
	mISDNSIGNAL_VOLUME,
	mISDNSIGNAL_CONF,
	mISDNSIGNAL_NODATA,		/* no data required */
	mISDNSIGNAL_ECHO,
};

/* call-info structure CALLER */
struct caller_info {
	char id[32];			/* id of caller (user number) */
	char voip[64];			/* URI of voip (or gateway) */
	char intern[32];		/* internal id */
	char name[16];
	int isdn_port;			/* internal/external port (if call is isdn) */
	char interface[32];		/* interface name the call was from */
	int itype;			/* type of interface */
	int ntype;			/* type of number */
	int present;			/* presentation */
	int screen;			/* who provided the number */
	char display[84];		/* display information */
};

/* call-info structure DIALING */
struct dialing_info {
	char number[256];		/* number dialing (so far) */
	char interfaces[128];		/* interfaces for extenal calls */
	int itype;			/* type of interface */
	int ntype;			/* type of number */
	int sending_complete;		/* end of dialing */
};

/* call-info structure CONNECT */
struct connect_info {
	char id[32];			/* id of caller (user number) */
	char voip[64];			/* URI of voip (or gateway) */
	char intern[32];		/* internal id */
	char name[16];
	int isdn_port;			/* internal/external port (if call is isdn) */
	char interfaces[128];		/* interfaces for extenal calls */
	int itype;			/* type of interface */
	int ntype;			/* type of number */
	int present;			/* presentation */
	int screen;			/* who provided the number */
	char display[84];		/* display information */
};

/* call-info structure DISCONNECT */
struct disconnect_info {
	int cause;			/* reason for disconnect */
	int location;			/* disconnect location */
	char display[84];		/* optional display information */
};

/* call-info structure REDIR */
struct redir_info {
	char id[32];			/* id of caller (user number) */
	char voip[64];			/* host of voip (or gateway) */
	char intern[32];		/* internal id */
	int isdn_port;			/* internal/external port (if call is isdn) */
	int itype;			/* type of interface */
	int ntype;			/* type of number */
	int present;			/* presentation */
	int screen;			/* who provided the number */
	int reason;			/* reason for redirecing */
};

/* call-info structure BEARER */
struct bearer_info {
	int capability;			/* type of data */
	int mode;			/* circuit/packet */
	int codec;			/* alaw,ulaw,... */
};

/* call-info structure NOTIFY */
struct notify_info {
	int notify;			/* notifications (see INFO_NOTIFY_*) */
	char id[32];			/* redirection id (user number) */
	char voip[64];			/* host of voip (or gateway) */
	char intern[32];		/* internal id */
	int isdn_port;			/* internal/external port (if call is isdn) */
	int itype;			/* type of interface */
	int ntype;			/* type of number */
	int present;			/* redirection presentation */
	char display[84];		/* display information */
	int local;			/* if set, endpoints gets information about audio channel (open/close) */
};

/* call-info structure FACILITY */
struct facility_info {
	char data[256];			/* data info about facility */
	int len;			/* length of facility content */
};

/* call-info structure USERUSER */
struct useruser_info {
	int protocol;
	int len;
	unsigned char data[128];	/* user-user info (not a sting!)*/
};

/* call-info structure SETUP */ 
struct message_setup {
	int isdn_port; /* card number 1...n (only on calls from internal isdn port) */
	int port_type; /* type of port (only required if message is port -> epoint) */
	int dtmf; /* used to enabled dtmf dialing at setup state */
	int partyline; /* if set, call will be a conference room */
	struct caller_info callerinfo;		/* information about the caller */
	struct dialing_info dialinginfo;	/* information about dialing */
	struct redir_info redirinfo;		/* info on redirection (to the calling user) */
	struct bearer_info bearerinfo;		/* info on l2,l3 capability */
	struct useruser_info useruser;		/* user-user */
};

/* call-info structure PARK */
struct park_info {
	char callid[8];
	int len;
};

/* DATA */
struct param_data {
	unsigned char data[512]; /* audio/hdlc data */
	int len; /* audio/hdlc data */
	int compressed; /* 0 for law-data, 1 for 16-bit data */
	unsigned long port_id; /* to identify the source of this data */
	int port_type; /* type of the source's port  */
};

struct param_play {
	char file[512]; /* file name */
	int offset; /* offset to start file at (in seconds) */
};

struct param_tone {
	char dir[128]; /* directory */
	char name[128]; /* file name */
};

struct param_counter {
	int current; /* current counter in seconds */
	int max; /* total size of file (0=no info) */
};

struct param_mISDNsignal {
	int message;
	int rxvol;
	int txvol;
	int conf;
	int nodata;
	int tone;
	int echo;
};

/* encryption control structure CRYPT */
struct param_crypt {
	int type; /* see messages in crypt.h */
	int len;
	unsigned char data[512+32]; /* a block of 512 byte + some overhead */
};

/* structure of message parameter */
union parameter {
	struct param_tone tone; /* MESSAGE_TONE */
	char dtmf; /* MESSAGE_DTMF */
	struct message_setup setup; /* MESSAGE_SETUP */
	struct dialing_info information; /* MESSAGE_INFO */
	struct connect_info connectinfo; /* CONNECT INFO */
	struct disconnect_info disconnectinfo; /* DISCONNECT INFO */
	struct notify_info notifyinfo; /* some notifications */
	struct facility_info facilityinfo; /* some notifications */
	struct park_info parkinfo; /* MESSAGE_SUSPEND, MESSAGE_RESUME */
	int state; /* MESSAGE_TIMEOUT */
	int knock; /* MESSAGE_KNOCK 0=off !0=on */
	int channel; /* MESSAGE_CHANNEL see RELATION_CHANNEL_* (call.h) */
	struct param_data data; /* MESSAGE_DATA */
	struct param_play play; /* MESSAGE_VBOX_PLAY */
	int speed; /* MESSAGE_VBOX_PLAY_SPEED */
	struct param_counter counter; /* MESSAGE_TONE_COUNTER */
	struct param_mISDNsignal mISDNsignal; /* MESSAGE_mISDNSIGNAL */
	struct extension ext; /* tell port about extension information */
	struct param_crypt crypt; /* MESSAGE_CRYPT */
};

enum { /* message flow */
	PORT_TO_EPOINT,
	EPOINT_TO_CALL,
	CALL_TO_EPOINT,
	EPOINT_TO_PORT,
};

/* message structure */
struct message {
	struct message *next;
	int type; /* type of message */
	int flow; /* from where to where */
	unsigned long id_from; /* in case of flow==PORT_TO_EPOINT: id_from is the port's serial, id_to is the epoint's serial */
	unsigned long id_to;
	union parameter param;
};

enum { /* messages between entities */
	MESSAGE_NONE,		/* no message */
	MESSAGE_TONE,		/* set information tone (to isdn port) */
	MESSAGE_DTMF,		/* dtmf digit (from isdn port) */
	MESSAGE_mISDNSIGNAL,	/* special mixer command (down to isdn port) */

	MESSAGE_SETUP,		/* setup message */
	MESSAGE_INFORMATION,	/* additional digit information */
	MESSAGE_OVERLAP,	/* call accepted, send more information */
	MESSAGE_PROCEEDING,	/* proceeding */
	MESSAGE_ALERTING,	/* ringing */
	MESSAGE_CONNECT,	/* connect */
	MESSAGE_DISCONNECT,	/* disconnect with cause */
	MESSAGE_RELEASE,	/* release with cause */
	MESSAGE_TIMEOUT,	/* protocol state has timed out (port->epoint) */
	MESSAGE_NOTIFY,		/* used to send progress and notify infos */
	MESSAGE_FACILITY,	/* used to facility infos, like aocd */

	MESSAGE_SUSPEND,	/* suspend port */
	MESSAGE_RESUME,		/* resume port */

	MESSAGE_CHANNEL,	/* set status of audio path to endpoint (to call, audio is also set) */
	MESSAGE_REMOTE_AUDIO,	/* tell remote to set audio status */
	MESSAGE_PATTERN,	/* pattern information tones available */
	MESSAGE_NOPATTERN,	/* pattern information tones unavailable */
	MESSAGE_CRYPT,		/* encryption message */

	MESSAGE_DATA,		/* audio/hdlc data */

	MESSAGE_VBOX_PLAY,	/* play recorded file */
	MESSAGE_VBOX_PLAY_SPEED,/* change speed of file */
	MESSAGE_VBOX_TONE,	/* set answering VBOX tone */
	MESSAGE_TONE_COUNTER,	/* tone counter (for VBOX tone use) */
	MESSAGE_TONE_EOF,	/* tone is end of file */
	MESSAGE_VBOX_RECORD,	/* tell endpoint to start recording */
};

#define MESSAGES static const char *messages_txt[] = { \
	"MESSAGE_NONE", \
	"MESSAGE_TONE", \
	"MESSAGE_DTMF", \
	"MESSAGE_mISDNSIGNAL", \
	"MESSAGE_SETUP", \
	"MESSAGE_INFORMATION", \
	"MESSAGE_OVERLAP", \
	"MESSAGE_PROCEEDING", \
	"MESSAGE_ALERTING", \
	"MESSAGE_CONNECT", \
	"MESSAGE_DISCONNECT", \
	"MESSAGE_RELEASE", \
	"MESSAGE_TIMEOUT", \
	"MESSAGE_NOTIFY", \
	"MESSAGE_FACILITY", \
	"MESSAGE_SUSPEND", \
	"MESSAGE_RESUME", \
	"MESSAGE_CHANNEL", \
	"MESSAGE_REMOTE_AUDIO", \
	"MESSAGE_PATTERN", \
	"MESSAGE_NOPATTERN", \
	"MESSAGE_CRYPT", \
	"MESSAGE_DATA", \
	"MESSAGE_VBOX_PLAY", \
	"MESSAGE_VBOX_PLAY_SPEED", \
	"MESSAGE_VBOX_TONE", \
	"MESSAGE_TONE_COUNTER", \
	"MESSAGE_TONE_EOF", \
	"MESSAGE_VBOX_RECORD", \
};


struct message *message_create(int id_from, int id_to, int flow, int type);
void message_put(struct message *message);
struct message *message_get(void);
void message_free(struct message *message);


#endif
